package ru.yandex.direct.api.v5.entity.campaigns.validation

import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignSetting
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignStrategy
import com.yandex.direct.api.v5.campaigns.MobileAppCampaignUpdateItem
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.entity.campaigns.converter.toContextLimit
import ru.yandex.direct.api.v5.entity.campaigns.converter.toMobileAppCampaignExternalStrategy
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsRequestValidator.validateNetworkDefaultLimitPercent
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Function

object MobileAppCampaignUpdateRequestValidator {

    private val STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE =
        mapOf<MobileAppCampaignSearchStrategyTypeEnum, Function<MobileAppCampaignSearchStrategy, Any>>(
            MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                Function { it.wbMaximumClicks },
            MobileAppCampaignSearchStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS to
                Function { it.wbMaximumAppInstalls },
            MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPC to
                Function { it.averageCpc },
            MobileAppCampaignSearchStrategyTypeEnum.AVERAGE_CPI to
                Function { it.averageCpi },
            MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                Function { it.weeklyClickPackage },
            MobileAppCampaignSearchStrategyTypeEnum.PAY_FOR_INSTALL to
                Function { it.payForInstall })

    private val STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
        setOf<MobileAppCampaignSearchStrategyTypeEnum>()

    private val STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE =
        mapOf<MobileAppCampaignNetworkStrategyTypeEnum, Function<MobileAppCampaignNetworkStrategy, Any>>(
            MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT to
                Function { it.networkDefault },
            MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                Function { it.wbMaximumClicks },
            MobileAppCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_APP_INSTALLS to
                Function { it.wbMaximumAppInstalls },
            MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPC to
                Function { it.averageCpc },
            MobileAppCampaignNetworkStrategyTypeEnum.AVERAGE_CPI to
                Function { it.averageCpi },
            MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                Function { it.weeklyClickPackage },
            MobileAppCampaignNetworkStrategyTypeEnum.PAY_FOR_INSTALL to
                Function { it.payForInstall })

    private val STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
        setOf(MobileAppCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT)

    fun validateMobileAppCampaign(
        item: MobileAppCampaignUpdateItem,
        oldCampaign: MobileContentCampaign?,
    ): ValidationResult<MobileAppCampaignUpdateItem, DefectType> {
        val vb = ItemValidationBuilder.of<MobileAppCampaignUpdateItem, DefectType>(item)
        vb.apply {
            item(item.biddingStrategy, "BiddingStrategy")
                .checkBy(
                    {
                        validateCampaignStrategy(
                            it,
                            oldCampaign!!)
                    },
                    When.notNull()
                )
            list(item.settings, "Settings")
                .weakCheckEach { setting: MobileAppCampaignSetting ->
                    CampaignsCommonRequestValidator.settingIsSupported(setting.option)
                }
                .weakCheckEach { setting: MobileAppCampaignSetting ->
                     settingIsConsistentWithStrategy(setting, item.biddingStrategy, oldCampaign)
                }
        }
        return vb.result
    }

    private fun validateCampaignStrategy(
        strategy: MobileAppCampaignStrategy,
        campaignFromDb: MobileContentCampaign
    ): ValidationResult<MobileAppCampaignStrategy, DefectType>? {
        val oldStrategy = toMobileAppCampaignExternalStrategy(campaignFromDb.strategy, campaignFromDb.contextLimit)
        val vb = ItemValidationBuilder.of<MobileAppCampaignStrategy, DefectType>(strategy)
        vb.apply {
            item(strategy.search, "Search")
                .check(::validateSearchStrategyConsistent, When.notNull())
                .check(
                    ::validateSearchStrategyTypeIsSupported,
                    When.isValidAnd(When.notNull())
                )
            item(strategy.network, "Network")
                .check(::validateNetworkStrategyConsistent, When.notNull())
                .check(
                    ::validateNetworkStrategyTypeIsSupported,
                    When.isValidAnd(When.notNull())
                )
                .check({ validateNetworkDefaultLimitPercent(it.toContextLimit()) },
                    When.isValidAnd(When.notNull()))
            check({ validateStrategyTypesAreCompatible(it, oldStrategy) }, When.isValid())
            weakCheck({ limitPercentIsConsistentWithStrategy(it, campaignFromDb) }, When.isValid())
        }
        return vb.result
    }

    private fun validateSearchStrategyConsistent(searchStrategy: MobileAppCampaignSearchStrategy) =
        CampaignsRequestValidator.validateStrategyConsistent(
            searchStrategy,
            searchStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE,
            STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE,
            true
        )

    private fun validateNetworkStrategyConsistent(networkStrategy: MobileAppCampaignNetworkStrategy) =
        CampaignsRequestValidator.validateStrategyConsistent(
            networkStrategy,
            networkStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE,
            STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE,
            false
        )

    private fun validateSearchStrategyTypeIsSupported(
        searchStrategy: MobileAppCampaignSearchStrategy
    ) = when (searchStrategy.biddingStrategyType) {
        MobileAppCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH ->
            CampaignDefectTypes.impressionsBelowSearchStrategyIsNotSupported()
        MobileAppCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
            CampaignDefectTypes.weeklyClickPackageNotSupported()
        else ->
            null
    }

    private fun validateNetworkStrategyTypeIsSupported(
        networkStrategy: MobileAppCampaignNetworkStrategy
    ) = when (networkStrategy.biddingStrategyType) {
        MobileAppCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
            CampaignDefectTypes.weeklyClickPackageNotSupported()
        else ->
            null
    }

    private fun validateStrategyTypesAreCompatible(
        strategy: MobileAppCampaignStrategy,
        oldStrategy: MobileAppCampaignStrategy
    ): DefectType? {
        val searchStrategyType = strategy.search?.biddingStrategyType ?: oldStrategy.search.biddingStrategyType!!
        val networkStrategyType = strategy.network?.biddingStrategyType ?: oldStrategy.network.biddingStrategyType!!
        val compatibleNetworkTypes = getNetworkStrategyTypesCompatibleWith(searchStrategyType)
        return if (!compatibleNetworkTypes.contains(networkStrategyType)) {
            CampaignDefectTypes.strategiesAreNotCompatible()
        } else null
    }
}
