package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.SmartCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.SmartCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.SmartCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.SmartCampaignStrategyAdd;
import com.yandex.direct.api.v5.campaigns.SmartCampaignStrategyAddBase;
import com.yandex.direct.api.v5.campaigns.StrategyNetworkDefaultAdd;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.strategiesAreNotCompatible;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.SmartCampaignValidatorKt.getNetworkStrategyTypesCompatibleWith;

@ParametersAreNonnullByDefault
public class SmartCampaignAddRequestValidator {

    private static final Map<SmartCampaignSearchStrategyTypeEnum, Function<SmartCampaignSearchStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE = Map.of(
            SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getAverageCpcPerCampaign,

            SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPC_PER_FILTER,
            SmartCampaignStrategyAddBase::getAverageCpcPerFilter,

            SmartCampaignSearchStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getAverageCpaPerCampaign,

            SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getPayForConversionPerCampaign,

            SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER,
            SmartCampaignStrategyAddBase::getPayForConversionPerFilter,

            SmartCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
            SmartCampaignStrategyAddBase::getWbMaximumConversionRate,

            SmartCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR,
            SmartCampaignStrategyAddBase::getPayForConversionCrr,

            SmartCampaignSearchStrategyTypeEnum.AVERAGE_ROI,
            SmartCampaignStrategyAddBase::getAverageRoi,

            SmartCampaignSearchStrategyTypeEnum.AVERAGE_CRR,
            SmartCampaignStrategyAddBase::getAverageCrr
    );

    private static final Set<SmartCampaignSearchStrategyTypeEnum> STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    private static final Map<SmartCampaignNetworkStrategyTypeEnum, Function<SmartCampaignNetworkStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE = Map.of(
            SmartCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT,
            SmartCampaignNetworkStrategyAdd::getNetworkDefault,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getAverageCpcPerCampaign,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPC_PER_FILTER,
            SmartCampaignStrategyAddBase::getAverageCpcPerFilter,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getAverageCpaPerCampaign,

            SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_CAMPAIGN,
            SmartCampaignStrategyAddBase::getPayForConversionPerCampaign,

            SmartCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION_PER_FILTER,
            SmartCampaignStrategyAddBase::getPayForConversionPerFilter,

            SmartCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
            SmartCampaignStrategyAddBase::getWbMaximumConversionRate,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CPA_PER_FILTER,
            SmartCampaignStrategyAddBase::getAverageCpaPerFilter,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_ROI,
            SmartCampaignStrategyAddBase::getAverageRoi,

            SmartCampaignNetworkStrategyTypeEnum.AVERAGE_CRR,
            SmartCampaignStrategyAddBase::getAverageCrr
    );

    private static final Set<SmartCampaignNetworkStrategyTypeEnum> STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
            Set.of(SmartCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT);

    public static ValidationResult<SmartCampaignAddItem, DefectType> validateSmartCampaign(
            SmartCampaignAddItem smartCampaignAdd) {
        ItemValidationBuilder<SmartCampaignAddItem, DefectType> vb = ItemValidationBuilder.of(smartCampaignAdd);

        vb.item(smartCampaignAdd.getBiddingStrategy(), "BiddingStrategy")
                .checkBy(SmartCampaignAddRequestValidator::validateCampaignStrategy);

        return vb.getResult();
    }

    public static ValidationResult<SmartCampaignStrategyAdd, DefectType> validateCampaignStrategy(
            SmartCampaignStrategyAdd strategy) {
        ItemValidationBuilder<SmartCampaignStrategyAdd, DefectType> vb = ItemValidationBuilder.of(strategy);

        vb.check(SmartCampaignAddRequestValidator::validateSearchStrategyConsistent)
                .check(SmartCampaignAddRequestValidator::validateNetworkStrategyConsistent)
                .check(SmartCampaignAddRequestValidator::validateStrategyTypesAreCompatible, When.isValid())
                .check(SmartCampaignAddRequestValidator::validateContextStrategy, When.isValid())
                .weakCheck(SmartCampaignValidatorKt::limitPercentIsConsistentWithStrategy, When.isValid());

        return vb.getResult();
    }

    @Nullable
    private static DefectType validateSearchStrategyConsistent(SmartCampaignStrategyAdd strategy) {
        SmartCampaignSearchStrategyAdd searchStrategy = strategy.getSearch();
        SmartCampaignSearchStrategyTypeEnum searchStrategyType = searchStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(searchStrategy, searchStrategyType,
                STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE, true);
    }

    @Nullable
    private static DefectType validateNetworkStrategyConsistent(SmartCampaignStrategyAdd strategy) {
        SmartCampaignNetworkStrategyAdd networkStrategy = strategy.getNetwork();
        SmartCampaignNetworkStrategyTypeEnum networkStrategyType = networkStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(networkStrategy, networkStrategyType,
                STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE, false);
    }

    @Nullable
    private static DefectType validateStrategyTypesAreCompatible(SmartCampaignStrategyAdd strategy) {
        SmartCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();
        SmartCampaignNetworkStrategyTypeEnum networkStrategyType = strategy.getNetwork().getBiddingStrategyType();

        return getNetworkStrategyTypesCompatibleWith(searchStrategyType).contains(networkStrategyType) ?
                null :
                strategiesAreNotCompatible();
    }

    /**
     * в perl _validate_context_strategy
     */
    @Nullable
    private static DefectType validateContextStrategy(SmartCampaignStrategyAdd strategy) {
        Integer limitPercent = null;
        var strategyType = strategy.getNetwork().getBiddingStrategyType();
        if (strategyType == SmartCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT) {
            limitPercent = Optional.of(strategy)
                    .map(SmartCampaignStrategyAdd::getNetwork)
                    .map(SmartCampaignNetworkStrategyAdd::getNetworkDefault)
                    .map(StrategyNetworkDefaultAdd::getLimitPercent)
                    .orElse(null);
        }
        // Вообще в смартах NETWORK_DEFAULT может быть только если задана стратегия на сетях
        // Все стратегии на сетях в смартах - автобюджетные.
        // А для автобюджетных стратегий не можем задавать LimitPercent
        // LimitPercent - будет всё равно проигнорирован, так что можно было и не валидировать
        // но мы валидируем, просто чтобы было как в перле
        return CampaignsRequestValidator.validateNetworkDefaultLimitPercent(limitPercent);
    }

}
