package ru.yandex.direct.api.v5.entity.campaigns.validation;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaigns.StrategyNetworkDefaultAdd;
import com.yandex.direct.api.v5.campaigns.TextCampaignAddItem;
import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyAdd;
import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaigns.TextCampaignSetting;
import com.yandex.direct.api.v5.campaigns.TextCampaignStrategyAdd;
import com.yandex.direct.api.v5.campaigns.TextCampaignStrategyAddBase;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.impressionsBelowSearchStrategyIsNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.strategiesAreNotCompatible;
import static ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes.weeklyClickPackageNotSupported;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.TextCampaignValidatorKt.getNetworkStrategyTypesCompatibleWith;
import static ru.yandex.direct.api.v5.entity.campaigns.validation.TextCampaignValidatorKt.settingIsConsistentWithStrategy;

@ParametersAreNonnullByDefault
public class TextCampaignAddRequestValidator {

    private static final Map<TextCampaignSearchStrategyTypeEnum, Function<TextCampaignSearchStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE = Map.of(
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CPA, TextCampaignStrategyAddBase::getAverageCpa,
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CPC, TextCampaignStrategyAddBase::getAverageCpc,
            TextCampaignSearchStrategyTypeEnum.AVERAGE_ROI, TextCampaignStrategyAddBase::getAverageRoi,
            TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION, TextCampaignStrategyAddBase::getPayForConversion,
            TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR,
            TextCampaignStrategyAddBase::getPayForConversionCrr,
            TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS, TextCampaignStrategyAddBase::getWbMaximumClicks,
            TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
            TextCampaignStrategyAddBase::getWbMaximumConversionRate,
            TextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE, TextCampaignStrategyAddBase::getWeeklyClickPackage,
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CRR, TextCampaignStrategyAddBase::getAverageCrr
    );

    private static final Set<TextCampaignSearchStrategyTypeEnum> STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
            Set.of();

    private static final Map<TextCampaignNetworkStrategyTypeEnum, Function<TextCampaignNetworkStrategyAdd, Object>>
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE = Map.of(
            TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT, TextCampaignNetworkStrategyAdd::getNetworkDefault,
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPA, TextCampaignStrategyAddBase::getAverageCpa,
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPC, TextCampaignStrategyAddBase::getAverageCpc,
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CRR, TextCampaignStrategyAddBase::getAverageCrr,
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_ROI, TextCampaignStrategyAddBase::getAverageRoi,
            TextCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION, TextCampaignStrategyAddBase::getPayForConversion,
            TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS, TextCampaignStrategyAddBase::getWbMaximumClicks,
            TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE,
            TextCampaignStrategyAddBase::getWbMaximumConversionRate,
            TextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE, TextCampaignStrategyAddBase::getWeeklyClickPackage
    );

    private static final Set<TextCampaignNetworkStrategyTypeEnum> STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
            Set.of(TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT);

    public static ValidationResult<TextCampaignAddItem, DefectType> validateTextCampaign(
            TextCampaignAddItem textCampaignAdd) {
        ItemValidationBuilder<TextCampaignAddItem, DefectType> vb = ItemValidationBuilder.of(textCampaignAdd);

        vb.item(textCampaignAdd.getBiddingStrategy(), "BiddingStrategy")
                .checkBy(TextCampaignAddRequestValidator::validateCampaignStrategy);
        vb.list(textCampaignAdd.getSettings(), "Settings")
                .weakCheckEach((TextCampaignSetting setting) ->
                        CampaignsCommonRequestValidator.settingIsSupported(setting.getOption()))
                .weakCheckEach((TextCampaignSetting setting) ->
                        settingIsConsistentWithStrategy(setting, textCampaignAdd.getBiddingStrategy()));

        return vb.getResult();
    }

    private static ValidationResult<TextCampaignStrategyAdd, DefectType> validateCampaignStrategy(
            TextCampaignStrategyAdd strategy) {
        ItemValidationBuilder<TextCampaignStrategyAdd, DefectType> vb = ItemValidationBuilder.of(strategy);


        vb.check(TextCampaignAddRequestValidator::validateSearchStrategyConsistent)
                .check(TextCampaignAddRequestValidator::validateNetworkStrategyConsistent)
                .check(TextCampaignAddRequestValidator::validateSearchStrategyTypeIsSupported, When.isValid())
                .check(TextCampaignAddRequestValidator::validateNetworkStrategyTypeIsSupported, When.isValid())
                .check(TextCampaignAddRequestValidator::validateStrategyTypesAreCompatible, When.isValid())
                .check(TextCampaignAddRequestValidator::validateContextStrategy, When.isValid())
                .weakCheck(TextCampaignValidatorKt::limitPercentIsConsistentWithStrategy, When.isValid());

        return vb.getResult();
    }

    @Nullable
    private static DefectType validateSearchStrategyConsistent(TextCampaignStrategyAdd strategy) {
        TextCampaignSearchStrategyAdd searchStrategy = strategy.getSearch();
        TextCampaignSearchStrategyTypeEnum searchStrategyType = searchStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(searchStrategy, searchStrategyType,
                STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE, true);
    }

    @Nullable
    private static DefectType validateNetworkStrategyConsistent(TextCampaignStrategyAdd strategy) {
        TextCampaignNetworkStrategyAdd networkStrategy = strategy.getNetwork();
        TextCampaignNetworkStrategyTypeEnum networkStrategyType = networkStrategy.getBiddingStrategyType();
        return CampaignsRequestValidator.validateStrategyConsistent(networkStrategy, networkStrategyType,
                STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE, false);
    }

    @Nullable
    private static DefectType validateSearchStrategyTypeIsSupported(TextCampaignStrategyAdd strategy) {
        TextCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();

        if (searchStrategyType == TextCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH) {
            return impressionsBelowSearchStrategyIsNotSupported();
        }

        if (searchStrategyType == TextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }

        return null;
    }

    @Nullable
    private static DefectType validateNetworkStrategyTypeIsSupported(TextCampaignStrategyAdd strategy) {
        var strategyType = strategy.getNetwork().getBiddingStrategyType();

        if (strategyType == TextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE) {
            return weeklyClickPackageNotSupported();
        }
        return null;
    }

    @Nullable
    private static DefectType validateStrategyTypesAreCompatible(TextCampaignStrategyAdd strategy) {
        TextCampaignSearchStrategyTypeEnum searchStrategyType = strategy.getSearch().getBiddingStrategyType();
        TextCampaignNetworkStrategyTypeEnum networkStrategyType = strategy.getNetwork().getBiddingStrategyType();

        return getNetworkStrategyTypesCompatibleWith(searchStrategyType).contains(networkStrategyType) ?
                null :
                strategiesAreNotCompatible();
    }

    /**
     * в perl _validate_context_strategy
     */
    @Nullable
    private static DefectType validateContextStrategy(TextCampaignStrategyAdd strategy) {
        Integer limitPercent = null;
        var strategyType = strategy.getNetwork().getBiddingStrategyType();
        if (strategyType == TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT) {
            limitPercent = Optional.of(strategy)
                    .map(TextCampaignStrategyAdd::getNetwork)
                    .map(TextCampaignNetworkStrategyAdd::getNetworkDefault)
                    .map(StrategyNetworkDefaultAdd::getLimitPercent)
                    .orElse(null);
        }
        return CampaignsRequestValidator.validateNetworkDefaultLimitPercent(limitPercent);
    }

}
