package ru.yandex.direct.api.v5.entity.campaigns.validation

import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaigns.TextCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategy
import com.yandex.direct.api.v5.campaigns.TextCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaigns.TextCampaignSetting
import com.yandex.direct.api.v5.campaigns.TextCampaignStrategy
import com.yandex.direct.api.v5.campaigns.TextCampaignUpdateItem
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.entity.campaigns.converter.toContextLimit
import ru.yandex.direct.api.v5.entity.campaigns.converter.toTextCampaignExternalStrategy
import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsRequestValidator.validateNetworkDefaultLimitPercent
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.api.v5.validation.DefectTypes
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Function

object TextCampaignUpdateRequestValidator {

    private val STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE =
        mapOf<TextCampaignSearchStrategyTypeEnum, Function<TextCampaignSearchStrategy, Any>>(
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CPA to
                Function { it.averageCpa },
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CPC to
                Function { it.averageCpc },
            TextCampaignSearchStrategyTypeEnum.AVERAGE_ROI to
                Function { it.averageRoi },
            TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION to
                Function { it.payForConversion },
            TextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR to
                Function { it.payForConversionCrr },
            TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                Function { it.wbMaximumClicks },
            TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE to
                Function { it.wbMaximumConversionRate },
            TextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                Function { it.weeklyClickPackage },
            TextCampaignSearchStrategyTypeEnum.AVERAGE_CRR to
                Function { it.averageCrr })

    private val STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE =
        setOf<TextCampaignSearchStrategyTypeEnum>()

    private val STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE =
        mapOf<TextCampaignNetworkStrategyTypeEnum, Function<TextCampaignNetworkStrategy, Any>>(
            TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT to
                Function { it.networkDefault },
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPA to
                Function { it.averageCpa },
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CPC to
                Function { it.averageCpc },
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_CRR to
                Function { it.averageCrr },
            TextCampaignNetworkStrategyTypeEnum.AVERAGE_ROI to
                Function { it.averageRoi },
            TextCampaignNetworkStrategyTypeEnum.PAY_FOR_CONVERSION to
                Function { it.payForConversion },
            TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                Function { it.wbMaximumClicks },
            TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE to
                Function { it.wbMaximumConversionRate },
            TextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                Function { it.weeklyClickPackage })

    private val STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE =
        setOf(TextCampaignNetworkStrategyTypeEnum.NETWORK_DEFAULT)

    fun validateTextCampaign(
        item: TextCampaignUpdateItem,
        oldCampaign: TextCampaign?
    ): ValidationResult<TextCampaignUpdateItem, DefectType> {
        val vb = ItemValidationBuilder.of<TextCampaignUpdateItem, DefectType>(item)
        vb.apply {
            item(item.biddingStrategy, "BiddingStrategy")
                .checkBy(
                    {
                        validateCampaignStrategy(
                            it,
                            oldCampaign!!
                        )
                    },
                    When.notNull()
                )
            list(item.settings, "Settings")
                .weakCheckEach { setting: TextCampaignSetting ->
                    CampaignsCommonRequestValidator.settingIsSupported(setting.option)
                }
                .weakCheckEach { setting: TextCampaignSetting ->
                    settingIsConsistentWithStrategy(setting, item.biddingStrategy, oldCampaign)
                }
            item(item.priorityGoals?.value, "PriorityGoals")
                .check(CampaignsUpdateRequestValidator::priorityGoalsNotSupportedOperation)
            item(item.relevantKeywords?.value, "RelevantKeywords")
                .check { CampaignsUpdateRequestValidator.absentValueBudgetPercent(it, oldCampaign) }
            check(::optimizeGoalIdMustNotBeNegative)
        }
        return vb.result
    }

    private fun validateCampaignStrategy(
        strategy: TextCampaignStrategy,
        campaignFromDb: TextCampaign
    ): ValidationResult<TextCampaignStrategy, DefectType>? {
        val oldStrategy = toTextCampaignExternalStrategy(campaignFromDb.strategy, campaignFromDb.contextLimit)
        val vb = ItemValidationBuilder.of<TextCampaignStrategy, DefectType>(strategy)
        vb.apply {
            item(strategy.search, "Search")
                .check(::validateSearchStrategyConsistent, When.notNull())
                .check(
                    ::validateSearchStrategyTypeIsSupported,
                    When.isValidAnd(When.notNull())
                )
            item(strategy.network, "Network")
                .check(::validateNetworkStrategyConsistent, When.notNull())
                .check(
                    ::validateNetworkStrategyTypeIsSupported,
                    When.isValidAnd(When.notNull())
                )
                .check({ validateNetworkDefaultLimitPercent(it.toContextLimit()) },
                    When.isValidAnd(When.notNull()))
            check({ validateStrategyTypesAreCompatible(it, oldStrategy) }, When.isValid())
            weakCheck({ limitPercentIsConsistentWithStrategy(it, campaignFromDb) }, When.isValid())
            check(::goalIdNotSet)
        }
        return vb.result
    }

    private fun optimizeGoalIdMustNotBeNegative(item: TextCampaignUpdateItem): DefectType? {
        val ok = item.relevantKeywords?.value?.optimizeGoalId?.value?.let { it >= 0 }
        return if (ok == false) DefectTypes.fieldShouldBePositive("OptimizeGoalId") else null
    }

    private fun validateSearchStrategyConsistent(
        searchStrategy: TextCampaignSearchStrategy
    ) =
        CampaignsRequestValidator.validateStrategyConsistent(
            searchStrategy,
            searchStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE,
            STRATEGY_TYPES_SEARCH_WITH_OPTIONAL_STRUCTURE,
            true
        )

    private fun validateNetworkStrategyConsistent(
        networkStrategy: TextCampaignNetworkStrategy
    ) =
        CampaignsRequestValidator.validateStrategyConsistent(
            networkStrategy,
            networkStrategy.biddingStrategyType,
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE,
            STRATEGY_TYPES_NETWORK_WITH_OPTIONAL_STRUCTURE,
            false
        )

    private fun validateSearchStrategyTypeIsSupported(
        searchStrategy: TextCampaignSearchStrategy
    ) = when (searchStrategy.biddingStrategyType) {
        TextCampaignSearchStrategyTypeEnum.IMPRESSIONS_BELOW_SEARCH ->
            CampaignDefectTypes.impressionsBelowSearchStrategyIsNotSupported()
        TextCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
            CampaignDefectTypes.weeklyClickPackageNotSupported()
        else ->
            null
    }

    private fun validateNetworkStrategyTypeIsSupported(
        networkStrategy: TextCampaignNetworkStrategy
    ) = when (networkStrategy.biddingStrategyType) {
        TextCampaignNetworkStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
            CampaignDefectTypes.weeklyClickPackageNotSupported()
        else ->
            null
    }

    private fun validateStrategyTypesAreCompatible(
        strategy: TextCampaignStrategy,
        oldStrategy: TextCampaignStrategy
    ): DefectType? {
        val searchStrategyType = strategy.search?.biddingStrategyType ?: oldStrategy.search.biddingStrategyType!!
        val networkStrategyType = strategy.network?.biddingStrategyType ?: oldStrategy.network.biddingStrategyType!!
        val compatibleNetworkTypes = getNetworkStrategyTypesCompatibleWith(searchStrategyType)
        return if (!compatibleNetworkTypes.contains(networkStrategyType)) {
            CampaignDefectTypes.strategiesAreNotCompatible()
        } else null
    }

    private fun goalIdNotSet(strategy: TextCampaignStrategy): DefectType? {
        val searchStrategy: TextCampaignSearchStrategy? = strategy.search
        val networkStrategy: TextCampaignNetworkStrategy? = strategy.network
        val notOk =
            networkStrategy?.biddingStrategyType == TextCampaignNetworkStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE
                && networkStrategy.wbMaximumConversionRate != null
                && networkStrategy.wbMaximumConversionRate.goalId == null
                || searchStrategy?.biddingStrategyType == TextCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CONVERSION_RATE
                && searchStrategy.wbMaximumConversionRate != null
                && searchStrategy.wbMaximumConversionRate.goalId == null
        return if (notOk) CampaignDefectTypes.goalIdNotSet() else null
    }
}
