package ru.yandex.direct.api.v5.entity.campaignsext.converter;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.campaignsext.AddRequest;
import com.yandex.direct.api.v5.campaignsext.CampaignAddItem;
import com.yandex.direct.api.v5.general.ArrayOfInteger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.campaigns.validation.CampaignsApiValidationSignals;
import ru.yandex.direct.api.v5.entity.campaignsext.validation.CampaignsExtRequestValidator;
import ru.yandex.direct.api.v5.entity.campaignsext.validation.CampaignsExtRequestValidatorKt;
import ru.yandex.direct.api.v5.entity.campaignsext.validation.ContentPromotionCampaignAddRequestValidator;
import ru.yandex.direct.api.v5.entity.campaignsext.validation.OtherCampaignsAddRequestValidator;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignDataConverterKt.toCampaignSource;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class CampaignsExtAddRequestConverter {

    private final UserService userService;
    private final GeoTimezoneRepository geoTimezoneRepository;
    private final ApiAuthenticationSource authenticationSource;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final OtherCampaignsAddItemConverter otherCampaignsAddItemConverter;
    private final OtherCampaignsAddRequestValidator otherCampaignsAddRequestValidator;

    @Autowired
    public CampaignsExtAddRequestConverter(UserService userService,
                                           GeoTimezoneRepository geoTimezoneRepository,
                                           ApiAuthenticationSource authenticationSource,
                                           SspPlatformsRepository sspPlatformsRepository,
                                           OtherCampaignsAddItemConverter otherCampaignsAddItemConverter,
                                           OtherCampaignsAddRequestValidator otherCampaignsAddRequestValidator) {
        this.userService = userService;
        this.geoTimezoneRepository = geoTimezoneRepository;
        this.authenticationSource = authenticationSource;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.otherCampaignsAddItemConverter = otherCampaignsAddItemConverter;
        this.otherCampaignsAddRequestValidator = otherCampaignsAddRequestValidator;
    }

    public List<BaseCampaign> convert(Long chiefUid, AddRequest addRequest) {
        String userEmail = userService.getUserEmail(chiefUid);
        List<CampaignAddItem> campaigns = addRequest.getCampaigns();
        Map<String, GeoTimezone> timeZones = getTimeZones(campaigns);
        var campaignSource = toCampaignSource(authenticationSource.getRequestSource());
        var knownSsp = listToSet(sspPlatformsRepository.getAllSspPlatforms());
        return mapList(campaigns,
                item -> convert(item, chiefUid, userEmail, timeZones, campaignSource, knownSsp));
    }

    private Map<String, GeoTimezone> getTimeZones(List<CampaignAddItem> campaigns) {
        List<String> timeZones = campaigns.stream()
                .map(CampaignAddItem::getTimeZone)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return geoTimezoneRepository.getByTimeZones(timeZones);
    }

    private BaseCampaign convert(CampaignAddItem item,
                                 Long chiefUid,
                                 String userEmail,
                                 Map<String, GeoTimezone> timeZones,
                                 CampaignSource campaignSource,
                                 Set<String> knownSsp) {
        ValidationResult<CampaignAddItem, DefectType> vr = validate(item, timeZones);

        if (vr.hasAnyErrors()) {
            // кампании содержащие ошибки в ядро передавать не будем, а значит и конвертировать в модели ядра не нужно
            return new CampaignsApiValidationSignals.CampaignWrapper()
                    .withDefects(mapList(vr.flattenErrors(), DefectInfo::getDefect))
                    .withWarnings(mapList(vr.flattenWarnings(), DefectInfo::getDefect))
                    .withRealCampaign(null);
        }
        return convertCampaign(item, chiefUid, userEmail, timeZones, campaignSource, knownSsp)
                .withDefects(mapList(vr.flattenErrors(), DefectInfo::getDefect))
                .withWarnings(mapList(vr.flattenWarnings(), DefectInfo::getDefect));
    }

    private CampaignsApiValidationSignals.CampaignWrapper convertCampaign(
            CampaignAddItem item,
            Long chiefUid,
            String userEmail,
            Map<String, GeoTimezone> timeZones,
            CampaignSource campaignSource,
            Set<String> knownSsps) {
        if (item.getContentPromotionCampaign() != null) {
            return ContentPromotionCampaignAddItemConverter.convertContentPromotionCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else if (item.getCpmBannerCampaign() != null) {
            return otherCampaignsAddItemConverter.convertCpmBannerCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else if (item.getDynamicTextCampaign() != null) {
            return otherCampaignsAddItemConverter.convertDynamicTextCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else if (item.getMobileAppCampaign() != null) {
            return otherCampaignsAddItemConverter.convertMobileAppCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else if (item.getSmartCampaign() != null) {
            return otherCampaignsAddItemConverter.convertSmartCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else if (item.getTextCampaign() != null) {
            return otherCampaignsAddItemConverter.convertTextCampaign(item, chiefUid, userEmail,
                    timeZones,
                    campaignSource, knownSsps);
        } else {
            throw new UnsupportedOperationException("Unexpected campaign type");
        }
    }

    private ValidationResult<CampaignAddItem, DefectType> validate(
            CampaignAddItem item,
            Map<String, GeoTimezone> timeZones) {
        ItemValidationBuilder<CampaignAddItem, DefectType> vb = ItemValidationBuilder.of(item);

        validateCampaignAddItemFatalDefects(vb);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.checkBy(campaignAddItem -> otherCampaignsAddRequestValidator.validateCampaignAddItemNormalDefects(
                campaignAddItem, timeZones));

        validateCampaignAddItemNormalDefectsForContentPromotion(vb);

        vb.item(item.getContentPromotionCampaign(), "ContentPromotionCampaign")
                .checkBy(ContentPromotionCampaignAddRequestValidator::validateContentPromotionCampaign,
                        When.notNull());
        vb.item(item.getCpmBannerCampaign(), "CpmBannerCampaign")
                .checkBy(otherCampaignsAddRequestValidator::validateCpmBannerCampaign, When.notNull());
        vb.item(item.getDynamicTextCampaign(), "DynamicTextCampaign")
                .checkBy(otherCampaignsAddRequestValidator::validateDynamicTextCampaign,
                        When.notNull());
        vb.item(item.getMobileAppCampaign(), "MobileAppCampaign")
                .checkBy(otherCampaignsAddRequestValidator::validateMobileAppCampaign,
                        When.notNull());
        vb.item(item.getSmartCampaign(), "SmartCampaign")
                .checkBy(otherCampaignsAddRequestValidator::validateSmartCampaign, When.notNull());
        vb.item(item.getTextCampaign(), "TextCampaign")
                .checkBy(otherCampaignsAddRequestValidator::validateTextCampaign, When.notNull());

        return vb.getResult();
    }

    /**
     * Проверяет на дефекты, которые не позволяют начать валидацию CampaignAddItem
     */
    private void validateCampaignAddItemFatalDefects(ItemValidationBuilder<CampaignAddItem, DefectType> vb) {
        vb.check(CampaignsExtRequestValidator.validCampaignType());
    }

    /**
     * Валидирует данные за рамками структур "TextCampaign", "CpmBannerCampaign" etc. (в основном это дефекты
     * общие для всех типов кампаний).
     */
    private void validateCampaignAddItemNormalDefectsForContentPromotion(
            ItemValidationBuilder<CampaignAddItem, DefectType> vb) {
        vb
                .check(CampaignsExtRequestValidatorKt::contentPromotionCampaignHasNoExcludedSites);
    }

    public static List<Long> toMetrikaCounters(@Nullable ArrayOfInteger counterIds) {
        if (counterIds == null) {
            return emptyList();
        }
        return mapList(counterIds.getItems(), Integer::longValue);
    }
}
