package ru.yandex.direct.api.v5.entity.campaignsext.converter

import com.yandex.direct.api.v5.campaignsext.CampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.CampaignGetItem
import com.yandex.direct.api.v5.campaignsext.CampaignTypeGetEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignGetItem
import com.yandex.direct.api.v5.campaignsext.CpmBannerCampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.DynamicTextCampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.MobileAppCampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.SmartCampaignFieldEnum
import com.yandex.direct.api.v5.campaignsext.TextCampaignFieldEnum
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.EnumPropertyFilter
import ru.yandex.direct.api.v5.common.toArrayOfInteger
import ru.yandex.direct.api.v5.entity.campaigns.container.CampaignAnyFieldEnum
import ru.yandex.direct.api.v5.entity.campaigns.container.GetCampaignsContainer
import ru.yandex.direct.api.v5.entity.campaigns.converter.CampaignsGetResponseConverter
import ru.yandex.direct.api.v5.entity.campaigns.converter.toApiAttributionModel
import ru.yandex.direct.common.util.PropertyFilter
import ru.yandex.direct.core.entity.campaign.model.ContentPromotionCampaign

@Component
class CampaignsExtGetResponseConverter(
    propertyFilter: PropertyFilter,
    private val campaignsGetResponseConverter: CampaignsGetResponseConverter,
) {
    private val baseCampaignFilter =
        EnumPropertyFilter.from(CampaignFieldEnum::class.java, propertyFilter)
    private val contentPromotionCampaignFilter =
        EnumPropertyFilter.from(ContentPromotionCampaignFieldEnum::class.java, propertyFilter)

    fun massConvertResponseItems(
        containers: Collection<GetCampaignsContainer>,
    ): Map<Long, CampaignGetItem> {
        val originalResponses: Map<Long, CampaignGetItem> = campaignsGetResponseConverter
            .massConvertResponseItems(containers)
            .mapValues { (_, item) -> item.convertBean() }

        return originalResponses
            .onEach { (campaignId, item) ->
                val container = containers.first { it.campaign.id == campaignId }

                if (container.campaign is ContentPromotionCampaign) {
                    item.type = CampaignTypeGetEnum.CONTENT_PROMOTION_CAMPAIGN
                    item.contentPromotionCampaign = createContentPromotionCampaignGetItem(container)
                }
            }
            .also { result ->
                if (result.isNotEmpty()) {
                    // для всех кампаний запрашиваются одинаковые поля,
                    // поэтому без разницы, из какого контейнера их брать
                    val requestedFields = containers.first().requestedFields
                    filterProperties(result.values, requestedFields)
                }
            }
    }

    private fun createContentPromotionCampaignGetItem(
        container: GetCampaignsContainer,
    ): ContentPromotionCampaignGetItem =
        ContentPromotionCampaignGetItem().apply {
            val campaign = container.campaign as ContentPromotionCampaign
            val requestedFields = container.requestedFields

            if (CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_BIDDING_STRATEGY in requestedFields) {
                biddingStrategy = createContentPromotionGetStrategy(campaign)
            }

            if (CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_ATTRIBUTION_MODEL in requestedFields) {
                attributionModel = campaign.attributionModel.toApiAttributionModel()
            }

            if (CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_COUNTER_IDS in requestedFields) {
                counterIds =
                    ru.yandex.direct.api.v5.entity.campaigns.converter.FACTORY.createTextCampaignBaseCounterIds(
                        campaign.metrikaCounters
                            ?.ifEmpty { null }
                            ?.toArrayOfInteger()
                    )
            }

            if (CampaignAnyFieldEnum.CONTENT_PROMOTION_CAMPAIGN_SETTINGS in requestedFields) {
                settings = createContentPromotionCampaignSettings(campaign, container.clientUid)
            }
        }

    fun filterProperties(items: Collection<CampaignGetItem>, requestedFields: Set<CampaignAnyFieldEnum>) {
        val requestedFieldsByType = requestedFields.groupBy { it.extApiValue::class }

        // Названия полей из CampaignFieldEnum,
        // а также "TextCampaign", "SmartCampaign" и т.д.
        val propertyNames = mutableListOf<String>()

        if (ContentPromotionCampaignFieldEnum::class in requestedFieldsByType) {
            val contentPromotionItems = items.mapNotNull { it.contentPromotionCampaign }
            val fields = requestedFieldsByType.getValue(ContentPromotionCampaignFieldEnum::class)
                .map { it.extApiValue as ContentPromotionCampaignFieldEnum }
            contentPromotionCampaignFilter.filterProperties(contentPromotionItems, fields)
            propertyNames += "contentPromotionCampaign"
        }

        // Поля структур TextCampaign, SmartCampaign и т.д.
        // уже отфильтрованы в CampaignsGetResponseConverter
        if (CpmBannerCampaignFieldEnum::class in requestedFieldsByType) {
            propertyNames += "cpmBannerCampaign"
        }

        if (DynamicTextCampaignFieldEnum::class in requestedFieldsByType) {
            propertyNames += "dynamicTextCampaign"
        }

        if (MobileAppCampaignFieldEnum::class in requestedFieldsByType) {
            propertyNames += "mobileAppCampaign"
        }

        if (SmartCampaignFieldEnum::class in requestedFieldsByType) {
            propertyNames += "smartCampaign"
        }

        if (TextCampaignFieldEnum::class in requestedFieldsByType) {
            propertyNames += "textCampaign"
        }

        val baseCampaignPropertyNames = requestedFieldsByType[CampaignFieldEnum::class]
            .orEmpty()
            .map { baseCampaignFilter.enumToFieldMap.getValue(it.extApiValue as CampaignFieldEnum) }
        propertyNames += baseCampaignPropertyNames
        baseCampaignFilter.filterPropertiesByNames(items, propertyNames)
    }
}
