package ru.yandex.direct.api.v5.entity.campaignsext.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignNetworkStrategyTypeEnum;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategy;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyAdd;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyTypeEnum;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignStrategy;
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignStrategyAdd;

import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;

import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.defaultStrategyData;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.fromWeeklyClickPackage;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.toCampOptionsStrategy;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyAddConverter.toCampaignsAutobudget;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyUpdateConverter.fromAverageCpc;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyUpdateConverter.fromWbMaximumClicks;
import static ru.yandex.direct.api.v5.entity.campaignsext.converter.StrategyUpdateConverter.fromWeeklyClickPackage;

/**
 * У кампаний ContentPromotion стратегия в сетях может быть только SERVING_OFF
 * То есть кампания доступна только на поиске
 *
 * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Direct/Model/CampaignContentPromotion.pm?rev=r8857536#L45">
 * Direct::Model::CampaignContentPromotion
 * </a>
 */
@ParametersAreNonnullByDefault
public class ContentPromotionCampaignStrategyConverter {

    private static final Map<ContentPromotionCampaignSearchStrategyTypeEnum, StrategyName> STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE =
            ImmutableMap.<ContentPromotionCampaignSearchStrategyTypeEnum, StrategyName>builder()
                    .put(ContentPromotionCampaignSearchStrategyTypeEnum.AVERAGE_CPC,
                            StrategyName.AUTOBUDGET_AVG_CLICK)
                    .put(ContentPromotionCampaignSearchStrategyTypeEnum.HIGHEST_POSITION,
                            StrategyName.DEFAULT_)
                    .put(ContentPromotionCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS,
                            StrategyName.AUTOBUDGET)
                    .put(ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE,
                            StrategyName.AUTOBUDGET_WEEK_BUNDLE)
                    .build();

    private ContentPromotionCampaignStrategyConverter() {
    }

    public static DbStrategy toCampaignStrategy(ContentPromotionCampaignStrategyAdd apiStrategy) {
        var dbStrategy = new DbStrategy();

        ContentPromotionCampaignSearchStrategyTypeEnum searchStrategyType =
                apiStrategy.getSearch().getBiddingStrategyType();
        ContentPromotionCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        StrategyName strategyName;
        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        // доступна только на поиске
        dbStrategy.withPlatform(CampaignsPlatform.SEARCH);
        dbStrategy.withStrategy(toCampOptionsStrategy(strategyName));
        return dbStrategy;
    }

    public static DbStrategy toCampaignStrategy(ContentPromotionCampaignStrategy apiStrategy) {
        var dbStrategy = new DbStrategy();

        ContentPromotionCampaignSearchStrategyTypeEnum searchStrategyType =
                apiStrategy.getSearch().getBiddingStrategyType();
        ContentPromotionCampaignNetworkStrategyTypeEnum networkStrategyType =
                apiStrategy.getNetwork().getBiddingStrategyType();

        StrategyName strategyName;
        if (STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.containsKey(searchStrategyType)) {
            strategyName = STRATEGY_NAME_BY_SEARCH_STRATEGY_TYPE.get(searchStrategyType);
            dbStrategy.withStrategyName(strategyName);
            dbStrategy.withAutobudget(toCampaignsAutobudget(strategyName));
            dbStrategy.withStrategyData(fromSearchStrategyData(searchStrategyType, apiStrategy.getSearch()));
        } else {
            throw new IllegalStateException(String.format("Strategy is not defined (search: %s, network %s)",
                    searchStrategyType, networkStrategyType));
        }

        // доступна только на поиске
        dbStrategy.withPlatform(CampaignsPlatform.SEARCH);
        dbStrategy.withStrategy(toCampOptionsStrategy(strategyName));
        return dbStrategy;
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(ContentPromotionCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       ContentPromotionCampaignSearchStrategyAdd strategy) {
        switch (searchStrategyType) {
            case HIGHEST_POSITION:
                return defaultStrategyData();
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            default:
                return null;
        }
    }

    @Nullable
    private static StrategyData fromSearchStrategyData(ContentPromotionCampaignSearchStrategyTypeEnum searchStrategyType,
                                                       ContentPromotionCampaignSearchStrategy strategy) {
        switch (searchStrategyType) {
            case HIGHEST_POSITION:
                return defaultStrategyData();
            case AVERAGE_CPC:
                return fromAverageCpc(strategy.getAverageCpc());
            case WB_MAXIMUM_CLICKS:
                return fromWbMaximumClicks(strategy.getWbMaximumClicks());
            case WEEKLY_CLICK_PACKAGE:
                return fromWeeklyClickPackage(strategy.getWeeklyClickPackage());
            default:
                return null;
        }
    }
}
