package ru.yandex.direct.api.v5.entity.campaignsext.converter

import com.yandex.direct.api.v5.campaignsext.CampaignUpdateItem
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategy
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSetting
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSettingsEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignStrategy
import com.yandex.direct.api.v5.general.YesNoEnum
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import ru.yandex.direct.api.v5.common.buildModelChanges
import ru.yandex.direct.api.v5.common.processJaxbElement
import ru.yandex.direct.api.v5.entity.campaigns.container.UpdateCampaignsConvertedRequest
import ru.yandex.direct.api.v5.entity.campaigns.converter.copyPropertiesExceptNull
import ru.yandex.direct.api.v5.entity.campaigns.converter.processAttributionModel
import ru.yandex.direct.api.v5.entity.campaigns.converter.processCheckPositionInterval
import ru.yandex.direct.api.v5.entity.campaigns.converter.processCommonFields
import ru.yandex.direct.api.v5.entity.campaigns.converter.processDailyBudget
import ru.yandex.direct.api.v5.entity.campaigns.converter.processFavoriteForIds
import ru.yandex.direct.api.v5.entity.campaigns.converter.processHasExtendedGeoTargeting
import ru.yandex.direct.api.v5.entity.campaigns.converter.processMetrikaCounters
import ru.yandex.direct.api.v5.entity.campaigns.converter.processStrategy
import ru.yandex.direct.api.v5.entity.campaigns.converter.processUseCurrentRegion
import ru.yandex.direct.api.v5.entity.campaigns.converter.processUseRegularRegion
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTimeTargeting
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.ContentPromotionCampaign
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.feature.FeatureName

@Component
class ContentPromotionCampaignUpdateItemConverter @Autowired constructor(
    private val featureService: FeatureService,
    private val campaignExtToCampaignConverter: CampaignExtToCampaignConverter
) : CampaignExtUpdateItemConverter<ContentPromotionCampaign> {

    override fun convert(
        item: CampaignUpdateItem,
        uidAndClientId: UidAndClientId,
        timeZones: Map<String, GeoTimezone>,
        knownSsps: Set<String>,
        campaignFromDb: BaseCampaign?
    ): UpdateCampaignsConvertedRequest<ContentPromotionCampaign> {
        check(campaignFromDb == null || campaignFromDb is ContentPromotionCampaign)
        val itemNotExt = campaignExtToCampaignConverter.convertUpdateItem(item)
        val contentPromotionItem = item.contentPromotionCampaign

        val settings = contentPromotionItem?.settings.orEmpty()
            .associateBy(ContentPromotionCampaignSetting::getOption) { it.value == YesNoEnum.YES }

        val modelChanges = buildModelChanges<ContentPromotionCampaign>(item.id) {
            val oldTimeTarget = campaignFromDb?.let { (it as CampaignWithTimeTargeting).timeTarget }
            val oldCommonCampaign = campaignFromDb?.let { (it as CommonCampaign) }
            processCommonFields(itemNotExt, timeZones, oldTimeTarget, oldCommonCampaign)
            processAttributionModel(contentPromotionItem?.attributionModel)
            processMetrikaCounters(contentPromotionItem?.counterIds)
            processDailyBudget(itemNotExt)
            processJaxbElement(item.negativeKeywords, ContentPromotionCampaign.MINUS_KEYWORDS) { it!!.items }
            processCheckPositionInterval(itemNotExt)

            val apiStrategy = contentPromotionItem?.biddingStrategy
            val newStrategy = apiStrategy?.let {
                updateStrategyWithOldValues(it, campaignFromDb as ContentPromotionCampaign)
                ContentPromotionCampaignStrategyConverter.toCampaignStrategy(it)
            }
            processStrategy(newStrategy)

            processHasExtendedGeoTargeting(
                settings[ContentPromotionCampaignSettingsEnum.ENABLE_AREA_OF_INTEREST_TARGETING]
            )

            val advancedGeoTargeting = featureService.isEnabledForClientId(uidAndClientId.clientId,
                FeatureName.ADVANCED_GEOTARGETING)
            if (advancedGeoTargeting) {
                processUseCurrentRegion(settings[ContentPromotionCampaignSettingsEnum.ENABLE_CURRENT_AREA_TARGETING])
                processUseRegularRegion(settings[ContentPromotionCampaignSettingsEnum.ENABLE_REGULAR_AREA_TARGETING])
            }
            processFavoriteForIds(uidAndClientId.uid, settings[ContentPromotionCampaignSettingsEnum.ADD_TO_FAVORITES])
        }

        return UpdateCampaignsConvertedRequest(
            modelChanges,
            requireServicing = settings[ContentPromotionCampaignSettingsEnum.REQUIRE_SERVICING]
        )
    }

    /**
     * Дозаполняет apiStrategy данными стратегии из oldCampaign, при условии, что biddingStrategyType одинаковый.
     */
    private fun updateStrategyWithOldValues(
        apiStrategy: ContentPromotionCampaignStrategy,
        oldCampaign: ContentPromotionCampaign
    ) {
        val oldStrategy = toContentPromotionCampaignExternalStrategy(oldCampaign.strategy)
        if (apiStrategy.search == null) {
            apiStrategy.search = oldStrategy.search
        } else if (apiStrategy.search.biddingStrategyType == oldStrategy.search.biddingStrategyType) {
            copyProperties(apiStrategy.search, oldStrategy.search)
            apiStrategy.search = oldStrategy.search
        }
        // ContentPromotionCampaign бывает только на поиске, в сетях всегда SERVING_OFF
        if (apiStrategy.network == null) {
            apiStrategy.network = oldStrategy.network
        }
    }

    @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
    private fun copyProperties(
        source: ContentPromotionCampaignSearchStrategy,
        dest: ContentPromotionCampaignSearchStrategy) {
        check(source.biddingStrategyType == dest.biddingStrategyType)
        when (source.biddingStrategyType) {
            ContentPromotionCampaignSearchStrategyTypeEnum.AVERAGE_CPC ->
                copyPropertiesExceptNull(source.averageCpc, dest.averageCpc)
            ContentPromotionCampaignSearchStrategyTypeEnum.HIGHEST_POSITION -> {
                // DEFAULT_STRATEGY_NAME, no properties
            }
            ContentPromotionCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS ->
                copyPropertiesExceptNull(source.wbMaximumClicks, dest.wbMaximumClicks)
            ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE ->
                copyPropertiesExceptNull(source.weeklyClickPackage, dest.weeklyClickPackage)
            ContentPromotionCampaignSearchStrategyTypeEnum.UNKNOWN -> {
                // not used
            }
        }
    }
}