package ru.yandex.direct.api.v5.entity.campaignsext.validation

import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignAddItem
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignNetworkStrategyAdd
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyAdd
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSetting
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSettingsEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignStrategyAdd
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Function

object ContentPromotionCampaignAddRequestValidator {
    private val STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE =
        mapOf(
            ContentPromotionCampaignSearchStrategyTypeEnum.AVERAGE_CPC to
                ContentPromotionCampaignSearchStrategyAdd::getAverageCpc,

            ContentPromotionCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                ContentPromotionCampaignSearchStrategyAdd::getWbMaximumClicks,

            ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                ContentPromotionCampaignSearchStrategyAdd::getWeeklyClickPackage,
        ).mapValues { (_, v) -> Function(v) }

    // Продвижение контента всегда выключено в сетях (кроме SERVING_OFF ничего быть не может)
    private val STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE =
        emptyMap<ContentPromotionCampaignNetworkStrategyTypeEnum, Function<ContentPromotionCampaignNetworkStrategyAdd, Any?>>()

    private val DEPRECATED_SETTINGS = emptySet<ContentPromotionCampaignSettingsEnum>()

    @JvmStatic
    fun validateContentPromotionCampaign(
        addItem: ContentPromotionCampaignAddItem
    ): ValidationResult<ContentPromotionCampaignAddItem, DefectType?> =
        ItemValidationBuilder.of<ContentPromotionCampaignAddItem, DefectType?>(addItem)
            .apply {
                item(addItem.biddingStrategy, "BiddingStrategy")
                    .checkBy { validateCampaignStrategy(it) }

                list(addItem.settings, "Settings")
                    .weakCheckEach(::settingIsSupported)
            }
            .result

    private fun validateCampaignStrategy(
        strategy: ContentPromotionCampaignStrategyAdd?
    ): ValidationResult<ContentPromotionCampaignStrategyAdd, DefectType?> =
        ItemValidationBuilder.of<ContentPromotionCampaignStrategyAdd, DefectType?>(strategy)
            .apply {
                check(::validateSearchStrategyConsistent)
                check(::validateNetworkStrategyConsistent)
                if (!result.hasAnyErrors()) {
                    check { validateSearchStrategyTypeIsSupported(it) }
                    check(::validateStrategyTypesAreCompatible)
                }
            }
            .result

    private fun settingIsSupported(setting: ContentPromotionCampaignSetting): DefectType? {
        return if (setting.option in DEPRECATED_SETTINGS) {
            CampaignDefectTypes.settingIsDeprecated(setting.option.toString())
        } else {
            null
        }
    }

    private fun validateSearchStrategyConsistent(
        strategy: ContentPromotionCampaignStrategyAdd,
    ): DefectType? {
        val searchStrategy = strategy.search
        val searchStrategyType = searchStrategy.biddingStrategyType
        return CampaignsExtRequestValidator.validateStrategyConsistent(
            searchStrategy, searchStrategyType,
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, true
        )
    }

    private fun validateNetworkStrategyConsistent(
        strategy: ContentPromotionCampaignStrategyAdd,
    ): DefectType? {
        val networkStrategy = strategy.network
        val networkStrategyType = networkStrategy.biddingStrategyType
        return CampaignsExtRequestValidator.validateStrategyConsistent(
            networkStrategy, networkStrategyType,
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, false
        )
    }

    private fun validateSearchStrategyTypeIsSupported(
        strategy: ContentPromotionCampaignStrategyAdd,
    ): DefectType? {
        val searchStrategyType = strategy.search.biddingStrategyType
        val isIncorrect = searchStrategyType == ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
        return if (isIncorrect) CampaignDefectTypes.weeklyClickPackageNotSupported() else null
    }

    private fun validateStrategyTypesAreCompatible(strategy: ContentPromotionCampaignStrategyAdd): DefectType? {
        val networkStrategyType = strategy.network.biddingStrategyType
        val isIncorrect = networkStrategyType != ContentPromotionCampaignNetworkStrategyTypeEnum.SERVING_OFF
        return if (isIncorrect) CampaignDefectTypes.strategiesAreNotCompatible() else null
    }
}
