package ru.yandex.direct.api.v5.entity.campaignsext.validation

import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignNetworkStrategy
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignNetworkStrategyTypeEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategy
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSearchStrategyTypeEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSetting
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignSettingsEnum
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignStrategy
import com.yandex.direct.api.v5.campaignsext.ContentPromotionCampaignUpdateItem
import ru.yandex.direct.api.v5.entity.campaigns.CampaignDefectTypes
import ru.yandex.direct.api.v5.validation.DefectType
import ru.yandex.direct.api.v5.validation.validateObject
import ru.yandex.direct.validation.builder.When
import java.util.function.Function

object ContentPromotionCampaignUpdateRequestValidator {

    private val STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE =
        mapOf(
            ContentPromotionCampaignSearchStrategyTypeEnum.AVERAGE_CPC to
                ContentPromotionCampaignSearchStrategy::getAverageCpc,

            ContentPromotionCampaignSearchStrategyTypeEnum.WB_MAXIMUM_CLICKS to
                ContentPromotionCampaignSearchStrategy::getWbMaximumClicks,

            ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE to
                ContentPromotionCampaignSearchStrategy::getWeeklyClickPackage,
        ).mapValues { (_, v) -> Function(v) }

    // Продвижение контента всегда выключено в сетях (кроме SERVING_OFF ничего быть не может)
    private val STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE =
        emptyMap<ContentPromotionCampaignNetworkStrategyTypeEnum,
            Function<ContentPromotionCampaignNetworkStrategy, Any?>>()

    private val DEPRECATED_SETTINGS = emptySet<ContentPromotionCampaignSettingsEnum>()

    @JvmStatic
    fun validateContentPromotionCampaign(
        updateItem: ContentPromotionCampaignUpdateItem
    ) = validateObject(updateItem) {
        item(updateItem.biddingStrategy, "BiddingStrategy")
            .checkBy({ validateCampaignStrategy(it) }, When.notNull())

        list(updateItem.settings, "Settings")
            .weakCheckEach(::settingIsSupported, When.notNull())
    }

    private fun validateCampaignStrategy(
        strategy: ContentPromotionCampaignStrategy
    ) = validateObject(strategy) {
        item(strategy.search, "Search")
            .check(::validateSearchStrategyConsistent, When.notNull())
        item(strategy.network, "Network")
            .check(::validateNetworkStrategyConsistent, When.notNull())
        if (!result.hasAnyErrors()) {
            item(strategy.search, "Search")
                .check(::validateSearchStrategyTypeIsSupported, When.notNull())
            // совместимость Search и Network можно не валидировать, т.к. в Network доступно только SERVING_OFF
        }
    }

    private fun settingIsSupported(setting: ContentPromotionCampaignSetting): DefectType? {
        return if (setting.option in DEPRECATED_SETTINGS) {
            CampaignDefectTypes.settingIsDeprecated(setting.option.toString())
        } else {
            null
        }
    }

    private fun validateSearchStrategyConsistent(
        searchStrategy: ContentPromotionCampaignSearchStrategy,
    ): DefectType? {
        val searchStrategyType = searchStrategy.biddingStrategyType
        return CampaignsExtRequestValidator.validateStrategyConsistent(
            searchStrategy, searchStrategyType,
            STRUCTURE_GETTER_BY_SEARCH_STRATEGY_TYPE, true
        )
    }

    private fun validateNetworkStrategyConsistent(
        networkStrategy: ContentPromotionCampaignNetworkStrategy,
    ): DefectType? {
        val networkStrategyType = networkStrategy.biddingStrategyType
        return CampaignsExtRequestValidator.validateStrategyConsistent(
            networkStrategy, networkStrategyType,
            STRUCTURE_GETTER_BY_NETWORK_STRATEGY_TYPE, false
        )
    }

    private fun validateSearchStrategyTypeIsSupported(
        searchStrategy: ContentPromotionCampaignSearchStrategy,
    ): DefectType? {
        val isIncorrect =
            searchStrategy.biddingStrategyType == ContentPromotionCampaignSearchStrategyTypeEnum.WEEKLY_CLICK_PACKAGE
        return if (isIncorrect) CampaignDefectTypes.weeklyClickPackageNotSupported() else null
    }
}