package ru.yandex.direct.api.v5.entity.changes.converter;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;

/**
 * Класс конвертирования таймстэмпа для сервиса Changes. Требуется именно такой формат: "yyyy-MM-dd'T'HH:mm:ss'Z'"
 *
 * @see <a href="https://yandex.ru/dev/direct/doc/ref-v5/changes/changes-docpage/">Сервис Changes</a>
 */
public class ChangesDateTimeConvertUtil {

    private static final int MOSCOW_TIMEZONE_DIFF_HOURS = 3;

    public static final DateTimeFormatter
            CHANGES_DATETIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss'Z'");

    /**
     * @param utcTimestampStr Строка, содержащая UTC-время и дату в формате "yyyy-MM-dd'T'HH:mm:ss'Z'"
     * @return Объект LocalDateTime с локальными (Московскими) датой и временем. Если utcTimestampStr == null, то
     * возвращается null.
     */
    public static LocalDateTime convertUtcTimestampToMoscowDateTime(String utcTimestampStr) {
        if (utcTimestampStr == null) {
            return null;
        }
        LocalDateTime utcDateTime = LocalDateTime.parse(utcTimestampStr, CHANGES_DATETIME_FORMATTER);
        utcDateTime = utcDateTime.plusHours(MOSCOW_TIMEZONE_DIFF_HOURS);
        return utcDateTime;
    }

    /**
     * Конвертировать время по Москве в строку с временем UTC.
     *
     * @param moscowLocalDateTime Московское время
     * @return Строка, содержащая UTC-время и дату в формате "yyyy-MM-dd'T'HH:mm:ss'Z'"
     */
    public static String convertMoscowDateTimeToUtcTimestamp(LocalDateTime moscowLocalDateTime) {
        LocalDateTime utcLocalDateTime = moscowLocalDateTime.minusHours(MOSCOW_TIMEZONE_DIFF_HOURS);
        return utcLocalDateTime.format(CHANGES_DATETIME_FORMATTER);
    }

    /**
     * @return Строка, содержащая текущее UTC-время и дату в формате "yyyy-MM-dd'T'HH:mm:ss'Z'"
     */
    public static String createNowUtcTimestampStr() {
        return LocalDateTime.now(ZoneOffset.UTC).format(CHANGES_DATETIME_FORMATTER);
    }
}
