package ru.yandex.direct.api.v5.entity.changes.delegate;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.changes.CheckCampaignsRequest;
import com.yandex.direct.api.v5.changes.CheckCampaignsResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.changes.converter.ChangesCommonConvertUtil;
import ru.yandex.direct.api.v5.entity.changes.converter.ChangesDateTimeConvertUtil;
import ru.yandex.direct.api.v5.entity.changes.validation.CheckCampaignsValidator;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.campaign.model.CampaignSourceUtils;
import ru.yandex.direct.core.entity.changes.model.CheckCampaignsIntResp;
import ru.yandex.direct.core.entity.changes.service.CheckCampaignsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

@Component
@ParametersAreNonnullByDefault
public class CheckCampaignsDelegate extends BaseApiServiceDelegate<
        CheckCampaignsRequest,
        CheckCampaignsResponse,
        LocalDateTime,
        CheckCampaignsIntResp> {

    private static final Logger logger = LoggerFactory.getLogger(CheckCampaignsDelegate.class);
    static final String USE_CAMP_AGGREGATED_LAST_CHANGE_HEADER = "useCampAggregatedLastChange";

    private final ShardHelper shardHelper;
    private final CheckCampaignsService checkCampaignsService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final EnvironmentType environmentType;
    private final PpcProperty<Boolean> useCampAggregatedLastChangeProp;

    @Autowired
    public CheckCampaignsDelegate(
            ApiAuthenticationSource auth,
            ShardHelper shardHelper,
            CheckCampaignsService checkCampaignsService,
            PpcPropertiesSupport ppcPropertiesSupport,
            EnvironmentType environmentType) {
        super(PathConverter.identity(), auth);
        this.shardHelper = shardHelper;
        this.checkCampaignsService = checkCampaignsService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.environmentType = environmentType;
        this.useCampAggregatedLastChangeProp = environmentType.isDevelopment() ?
                ppcPropertiesSupport.get(PpcPropertyNames.USE_CAMP_AGGREGATED_LASTCHANGE_PROPERTY) :
                ppcPropertiesSupport.get(PpcPropertyNames.USE_CAMP_AGGREGATED_LASTCHANGE_PROPERTY,
                        Duration.ofSeconds(300));
    }

    @Nullable
    @Override
    public ValidationResult<CheckCampaignsRequest, DefectType> validateRequest(CheckCampaignsRequest externalRequest) {
        return new CheckCampaignsValidator().apply(externalRequest);
    }

    @Override
    public LocalDateTime convertRequest(CheckCampaignsRequest externalRequest) {
        return ChangesDateTimeConvertUtil.convertUtcTimestampToMoscowDateTime(externalRequest.getTimestamp());
    }

    @Override
    public ApiResult<List<CheckCampaignsIntResp>> processRequest(LocalDateTime requestTimestamp) {
        ClientId clientId = auth.getSubclient().getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<CheckCampaignsIntResp> results = checkCampaignsService.getCampaignsChanges(clientId, shard,
                requestTimestamp, useCampAggregatedLastChange(),
                CampaignSourceUtils.getApi5VisibleSources(auth.getRequestSource()));

        return ApiResult.successful(results);
    }

    @Override
    public CheckCampaignsResponse convertResponse(ApiResult<List<CheckCampaignsIntResp>> result) {
        String timestamp = null;
        if (useCampAggregatedLastChange()) {
            int shard = shardHelper.getShardByClientIdStrictly(auth.getSubclient().getClientId());
            LocalDateTime localDateTime =
                    ppcPropertiesSupport.get(PpcPropertyNames.campAggregatedLastchangeHighBoundary(shard)).get();
            if (localDateTime != null) {
                timestamp = ChangesDateTimeConvertUtil.convertMoscowDateTimeToUtcTimestamp(localDateTime);
            }
        }
        if (timestamp == null) {
            timestamp = ChangesDateTimeConvertUtil.createNowUtcTimestampStr(); // текущее время - в респонс
        }
        return ChangesCommonConvertUtil.convertCheckCampaignsResult(result.getResult(), timestamp);
    }

    private boolean useCampAggregatedLastChange() {
        if (!environmentType.isProductionOrPrestable()) {
            String campAggregatedHeaderValue = HttpUtil.getRequest().getHeader(USE_CAMP_AGGREGATED_LAST_CHANGE_HEADER);
            if ("true".equalsIgnoreCase(campAggregatedHeaderValue)) {
                return true;
            }
            if ("false".equalsIgnoreCase(campAggregatedHeaderValue)) {
                return false;
            }
        }
        return useCampAggregatedLastChangeProp.getOrDefault(false);
    }
}
