package ru.yandex.direct.api.v5.entity.changes.delegate;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.changes.CheckDictionariesRequest;
import com.yandex.direct.api.v5.changes.CheckDictionariesResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.changes.converter.ChangesDateTimeConvertUtil;
import ru.yandex.direct.api.v5.entity.changes.validation.CheckDictionariesValidator;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.changes.model.CheckDictionariesIntResp;
import ru.yandex.direct.core.entity.changes.service.CheckDictionariesService;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesRepository;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.common.GeneralUtil.yesNoFromBool;

@Component
@ParametersAreNonnullByDefault
public class CheckDictionariesDelegate extends BaseApiServiceDelegate<
        CheckDictionariesRequest,
        CheckDictionariesResponse,
        LocalDateTime,
        CheckDictionariesIntResp> {

    private static final Logger logger = LoggerFactory.getLogger(CheckDictionariesDelegate.class);

    private final CheckDictionariesService checkDictionariesService;
    private final TargetingCategoriesRepository targetingCategoriesRepository;

    @Autowired
    public CheckDictionariesDelegate(
            ApiAuthenticationSource auth,
            CheckDictionariesService checkDictionariesService,
            TargetingCategoriesRepository targetingCategoriesRepository) {
        super(PathConverter.identity(), auth);
        this.checkDictionariesService = checkDictionariesService;
        this.targetingCategoriesRepository = targetingCategoriesRepository;
    }

    @Override
    public ValidationResult<CheckDictionariesRequest, DefectType> validateRequest(CheckDictionariesRequest externalRequest) {
        return new CheckDictionariesValidator().apply(externalRequest);
    }

    @Override
    public LocalDateTime convertRequest(CheckDictionariesRequest externalRequest) {
        return ChangesDateTimeConvertUtil.convertUtcTimestampToMoscowDateTime(externalRequest.getTimestamp());
    }

    @Override
    @Nullable
    public ApiResult<List<CheckDictionariesIntResp>> processRequest(@Nullable LocalDateTime requestTimestamp) {

        Boolean timeZoneWasChangedAfterOrEqualRequestTime = null;
        Boolean regionWasChangedAfterOrEqualRequestTime = null;
        Boolean targetingCategoriesChanged = null;

        if (requestTimestamp != null) {

            LocalDateTime geoRegionsUpdateTime = checkDictionariesService.getGeoRegionsUpdateTime();
            if (geoRegionsUpdateTime != null) {
                regionWasChangedAfterOrEqualRequestTime = !geoRegionsUpdateTime.isBefore(requestTimestamp);
            } else {
                regionWasChangedAfterOrEqualRequestTime = true; // на всякий случай, чтобы обновляли справочник
            }

            LocalDateTime geoTimezonesUpdateTime = checkDictionariesService.getGeoTimezonesUpdateTime();
            if (geoTimezonesUpdateTime != null) {
                timeZoneWasChangedAfterOrEqualRequestTime = !geoTimezonesUpdateTime.isBefore(requestTimestamp);
            } else {
                timeZoneWasChangedAfterOrEqualRequestTime = true; // на всякий случай, чтобы обновляли справочник
            }

            targetingCategoriesChanged = targetingCategoriesRepository.existsNewRecordsAfter(requestTimestamp);
        }

        return ApiResult.successful(List.of(
                new CheckDictionariesIntResp(
                        timeZoneWasChangedAfterOrEqualRequestTime,
                        regionWasChangedAfterOrEqualRequestTime,
                        targetingCategoriesChanged
                )
        ));
    }

    @Override
    public CheckDictionariesResponse convertResponse(ApiResult<List<CheckDictionariesIntResp>> result) {
        List<CheckDictionariesIntResp> list = result.getResult();
        if (!list.isEmpty()) {
            CheckDictionariesIntResp intResp = list.get(0);
            if (intResp != null) {
                CheckDictionariesResponse response = new CheckDictionariesResponse();
                if (intResp.getTimeZoneWasChanged() != null) {
                    response.setTimeZonesChanged(yesNoFromBool(intResp.getTimeZoneWasChanged()));
                }
                if (intResp.getRegionWasChanged() != null) {
                    response.setRegionsChanged(yesNoFromBool(intResp.getRegionWasChanged()));
                }
                if (intResp.getTimeZoneWasChanged() != null) {
                    response.setInterestsChanged(yesNoFromBool(intResp.getTargetingCategoriesChanged()));
                }
                response.setTimestamp(ChangesDateTimeConvertUtil.createNowUtcTimestampStr());
                return response;
            }
        }
        return null;
    }
}
