package ru.yandex.direct.api.v5.entity.clients.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.generalclients.ClientGetItem;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.function.Function;
import ru.yandex.direct.api.v5.entity.agencyclients.service.AgencySubclientGrants;
import ru.yandex.direct.api.v5.entity.agencyclients.service.BaseClientDataFetcher;
import ru.yandex.direct.api.v5.entity.agencyclients.service.ClientGetItemWriter;
import ru.yandex.direct.api.v5.entity.agencyclients.service.ClientGetItemWriters;
import ru.yandex.direct.api.v5.entity.agencyclients.service.ClientSubtype;
import ru.yandex.direct.api.v5.entity.agencyclients.service.ClientType;
import ru.yandex.direct.api.v5.entity.agencyclients.service.MultipleAgenciesException;
import ru.yandex.direct.api.v5.entity.agencyclients.service.RepresentativeInfo;
import ru.yandex.direct.api.v5.entity.agencyclients.service.RequestedField;
import ru.yandex.direct.core.entity.account.score.model.AccountScore;
import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.core.entity.client.model.ClientSpecialLimits;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientNdsService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.validation.CommonDefectTranslations;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.Representative;
import ru.yandex.direct.rbac.model.SubclientGrants;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.utils.CommonUtils.memoize;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class ClientDataFetcher {
    private static final Logger logger = LoggerFactory.getLogger(ClientDataFetcher.class);

    private final AgencyClientRelationService agencyClientRelationService;
    private final ClientService clientService;
    private final BaseClientDataFetcher dataFetcher;
    private final RbacService rbacService;
    private final UserService userService;
    private final FreelancerService freelancerService;
    private final ClientGetItemWriters writers;
    private final ClientNdsService clientNdsService;
    private final RbacClientsRelations rbacClientsRelations;

    @Autowired
    public ClientDataFetcher(AgencyClientRelationService agencyClientRelationService,
                             ClientService clientService, BaseClientDataFetcher dataFetcher,
                             RbacService rbacService, UserService userService,
                             FreelancerService freelancerService,
                             ClientGetItemWriters writers,
                             ClientNdsService clientNdsService,
                             RbacClientsRelations rbacClientsRelations) {
        this.agencyClientRelationService = agencyClientRelationService;
        this.clientService = clientService;
        this.dataFetcher = dataFetcher;
        this.rbacService = rbacService;
        this.userService = userService;
        this.freelancerService = freelancerService;
        this.writers = writers;
        this.clientNdsService = clientNdsService;
        this.rbacClientsRelations = rbacClientsRelations;
    }

    List<ClientGetItem> getData(Set<RequestedField> fieldNames, Long requestedUid, User operator,
                                User chiefSubclient, User subclient, Map<ClientId, Integer> clientUnitsLimits) {
        User user = userService.getUser(requestedUid);

        if (user == null) {
            logger.warn("Could not get user (uid: {})", requestedUid);
            return emptyList();
        }

        ClientId clientId = user.getClientId();
        @Nullable Client client = clientService.getClient(clientId);

        List<Client> clients = clientService.massGetClient(singletonList(clientId));
        Supplier<Map<ClientId, Client>> clientsSupplier = memoize(
                () -> listToMap(clients, c -> ClientId.fromLong(c.getId())));

        Supplier<Map<ClientId, ClientLimits>> clientsLimitsSupplier =
                memoize(() -> dataFetcher.getClientLimits(singletonList(clientId)));

        Supplier<Map<ClientId, ClientSpecialLimits>> clientsSpecialLimitsSupplier =
                memoize(() -> dataFetcher.getClientSpecialLimits(singletonList(clientId)));

        Supplier<Map<ClientId, List<RepresentativeInfo>>> representativesSupplier =
                memoize(() -> getRepresentatives(clientId, chiefSubclient, subclient));

        Supplier<Map<ClientId, Collection<AgencyClientRelation>>> agencyClientRelationsSupplier =
                memoize(() -> dataFetcher.getAgencyClientRelation(singletonList(clientId)));

        Supplier<Map<ClientId, Percent>> clientsNdsSupplier =
                memoize(() -> getClientsNds(agencyClientRelationsSupplier, client));

        Supplier<Map<ClientId, Currency>> clientsCurrenciesSupplier =
                memoize(() -> dataFetcher.getClientsCurrencies(singletonList(clientId)));

        Supplier<Map<ClientId, BigDecimal>> awaitingBonusesSupplier =
                memoize(() -> dataFetcher.getAwaitingBonuses(clients));

        Supplier<Map<ClientId, BigDecimal>> awaitingBonusesWithoutNdsSupplier = memoize(() -> dataFetcher
                .getAwaitingBonusesWithoutNds(clients, clientsNdsSupplier, clientsCurrenciesSupplier));

        Supplier<Map<ClientId, BigDecimal>> overdraftRestSupplier = memoize(() -> dataFetcher
                .getClientsOverdraftRest(clientsCurrenciesSupplier, clientsNdsSupplier, singletonList(clientId)));

        @Nullable ClientId agencyClientId = isAgency(operator) ? operator.getClientId() : null;
        Supplier<Map<ClientId, Boolean>> sharedAccountInfoSupplier =
                memoize(() -> dataFetcher.getSharedAccountInfo(
                        clientsCurrenciesSupplier, agencyClientRelationsSupplier, agencyClientId,
                        singletonList(clientId)));

        Supplier<List<String>> managedLoginsSupplier = memoize(() -> new ArrayList<>(
                userService.getChiefsLoginsByClientIds(
                        rbacClientsRelations.getMangedMccClientIds(clientId)
                ).values()));

        List<ClientGetItemWriter> fieldWriters = asList(
                writers.getClientIdWriter(),
                writers.getLoginWriter(),
                writers.getClientInfoWriter(),
                writers.getPhoneWriter(),
                writers.getCountryIdWriter(clientsSupplier),
                writers.getCreatedAtWriter(),
                writers.getAccountQualityWriter(accountScoreSupplier(clientId)),
                writers.getClientTypeWriter(clientTypesSupplier(operator, user, client)),
                writers.getClientSubtypeWriter(clientSubtypesSupplier(user)),
                writers.getArchivedWriter(unarchivedSupplier(operator, user, client)),
                writers.getNotificationWriter(smsPhoneSupplier(requestedUid)),
                writers.getSettingsWriter(clientsSupplier, sharedAccountInfoSupplier),
                writers.getRestrictionsWriter(clientsLimitsSupplier, clientsSpecialLimitsSupplier,
                        () -> clientUnitsLimits),
                writers.getGrantsWriter(subclientsGrantsSupplier(operator, user)),
                writers.getBonusesWriter(awaitingBonusesSupplier, awaitingBonusesWithoutNdsSupplier),
                writers.getRepresentativesWriter(representativesSupplier),
                writers.getCurrencyWriter(clientsCurrenciesSupplier),
                writers.getVatRateWriter(clientsNdsSupplier),
                writers.getManagedLoginsWriter(managedLoginsSupplier),
                writers.getOverdraftSumAvailableWriter(overdraftRestSupplier)
        );

        ClientGetItem clientGetItem = new ClientGetItem();
        fieldWriters.forEach(w -> w.write(fieldNames, clientGetItem, user));
        return singletonList(clientGetItem);
    }

    private Map<ClientId, Percent> getClientsNds(
            Supplier<Map<ClientId, Collection<AgencyClientRelation>>> agencyClientRelationsSupplier,
            @Nullable Client client) {
        if (client == null) {
            return Collections.emptyMap();
        }

        Map<ClientId, Collection<AgencyClientRelation>> relations = agencyClientRelationsSupplier.get();
        Collection<AgencyClientRelation> clientRelations =
                relations.getOrDefault(ClientId.fromLong(client.getId()), emptyList());
        List<AgencyClientRelation> activeRelations = clientRelations.stream()
                .filter(r -> !r.getArchived() && r.getBinded()).collect(toList());

        if (activeRelations.isEmpty() || client.getNonResident()) {
            // клиент без агенств или нерезидент
            return clientNdsService.massGetClientNds(singletonList(ClientId.fromLong(client.getId()))).stream()
                    .collect(toMap(c -> ClientId.fromLong(c.getClientId()), ClientNds::getNds));
        }

        if (activeRelations.size() > 1) {
            // не определяем НДС, если активных агенств несколько
            throw new MultipleAgenciesException(
                    CommonDefectTranslations.INSTANCE.notAllowedGetNdsClientMultipleAgencies());
        }

        // Для субклиентов-резидентов НДС равен НДС агенства
        ClientId agencyClientId = activeRelations.get(0).getAgencyClientId();
        ClientNds agencyNds = clientNdsService.getClientNds(agencyClientId);
        if (agencyNds == null) {
            logger.warn("Could not get NDS of agency (agency client id: {}", agencyClientId);
            return Collections.emptyMap();
        }
        return singletonMap(ClientId.fromLong(client.getId()), agencyNds.getNds());
    }

    private Supplier<Map<ClientId, List<AgencySubclientGrants>>> subclientsGrantsSupplier(User operator, User user) {
        return memoize(() -> getGrants(operator, user));
    }

    private Supplier<Map<Long, String>> smsPhoneSupplier(Long requestedUid) {
        return memoize(() -> dataFetcher.getSmsPhones(singletonList(requestedUid)));
    }

    private Supplier<Set<Long>> unarchivedSupplier(User operator, User user, @Nullable Client client) {
        return memoize(() -> getUnarchived(operator, user, client));
    }

    private Supplier<Map<Long, ClientType>> clientTypesSupplier(User operator, User user,
                                                                @Nullable Client client) {
        return memoize(() -> getClientTypes(operator, user, client));
    }

    private Supplier<Map<Long, ClientSubtype>> clientSubtypesSupplier(User user) {
        return memoize(() -> getClientSubtype(user));
    }

    private Supplier<Map<ClientId, AccountScore>> accountScoreSupplier(ClientId clientId) {
        return memoize(() -> dataFetcher.getAccountScore(singletonList(clientId)));
    }

    private Set<Long> getUnarchived(User operator, User user, @Nullable Client client) {
        if (isUnarchived(operator, user, client)) {
            return ImmutableSet.of(user.getClientId().asLong());
        } else {
            return emptySet();
        }
    }

    private boolean isUnarchived(User operator, User user, @Nullable Client client) {
        if (isAgency(operator) && !isAgency(user)) {
            return agencyClientRelationService.isUnArchivedAgencyClient(
                    operator.getClientId().asLong(), user.getClientId().asLong());
        }

        if (isAgency(user) || !rbacService.isUnderAgency(user.getUid())) {
            return !user.getStatusArch();
        }

        if (client != null && client.getAllowCreateScampBySubclient()) {
            // has freedom
            return !user.getStatusArch();
        }

        List<AgencyClientRelation> unArchivedRelations =
                agencyClientRelationService.getUnarchivedBindedAgencies(singletonList(user.getClientId()));

        return !unArchivedRelations.isEmpty();
    }

    private Map<Long, ClientType> getClientTypes(User operator, User user, @Nullable Client client) {
        if (isAgency(user)) {
            return singletonMap(user.getUid(), ClientType.AGENCY);
        }

        if (!rbacService.isUnderAgency(user.getUid())) {
            return singletonMap(user.getUid(), ClientType.CLIENT);
        }

        if (isAgency(operator) || (client != null && !client.getAllowCreateScampBySubclient())) {
            return singletonMap(user.getUid(), ClientType.SUBCLIENT);
        }

        return singletonMap(user.getUid(), ClientType.CLIENT);
    }

    private Map<Long, ClientSubtype> getClientSubtype(User user) {
        if (freelancerService.isFreelancer(user.getClientId())) {
            return singletonMap(user.getUid(), ClientSubtype.SPECIALIST);
        }

        return singletonMap(user.getUid(), ClientSubtype.NONE);
    }

    private Map<ClientId, List<AgencySubclientGrants>> getGrants(User operator, User user) {
        return singletonMap(user.getClientId(), getSubclientGrants(operator, user));
    }

    private List<AgencySubclientGrants> getSubclientGrants(User operator, User user) {
        if (!rbacService.isUnderAgency(user.getUid())) {
            return rbacService.getClientGrantsWithoutAgency(user.getClientId()).stream()
                    .map(e -> new AgencySubclientGrants(null, e))
                    .collect(Collectors.toList());
        }

        Collection<Long> agenciesUids;
        if (isAgency(operator)) {
            agenciesUids = Collections.singletonList(operator.getChiefUid());
        } else {
            agenciesUids = rbacService.getAgencyChiefsOfSubclient(user.getUid());
        }

        Collection<SubclientGrants> subclientGrants = rbacService.getSubclientGrants(agenciesUids,
                user.getClientId());

        Map<Long, Client> agencyByUid = clientService.massGetClientsByUids(agenciesUids);
        Function<SubclientGrants, String> getAgencyName = grants -> Optional.ofNullable(grants)
                .map(SubclientGrants::getAgencyUid)
                .map(agencyByUid::get)
                .map(Client::getName)
                .orElse(null);

        // При просмотре агентского клиента от агентства мы учитываем только это агентство, даже если обслуживание завершено.
        Set<ClientId> unbindedAgenciesIds = isAgency(operator) ?
                emptySet() : agencyClientRelationService.getUnbindedAgencies(user.getClientId());
        Set<Long> unbindedAgenciesUids = EntryStream.of(agencyByUid)
                .mapValues(Client::getId)
                .mapValues(ClientId::fromLong)
                .filterValues(unbindedAgenciesIds::contains)
                .keys()
                .toSet();

        return subclientGrants.stream()
                .filter(e -> !unbindedAgenciesUids.contains(e.getAgencyUid()))
                .map(e -> new AgencySubclientGrants(getAgencyName.apply(e), e))
                .collect(Collectors.toList());
    }

    private Map<ClientId, List<RepresentativeInfo>> getRepresentatives(ClientId clientId, User chiefSubclient,
                                                                       User subclient) {
        if (!Objects.equals(subclient.getUid(), chiefSubclient.getUid())) {
            // если субклиент не шеф, то он видит только шефа
            return singletonMap(subclient.getClientId(), singletonList(
                    new RepresentativeInfo(
                            Representative.createChief(chiefSubclient.getClientId(), chiefSubclient.getUid()),
                            chiefSubclient)));
        } else {
            Collection<Representative> representativesByClient = rbacService.getClientRepresentatives(clientId);
            List<Long> uids = mapList(representativesByClient, Representative::getUserId);
            Map<Long, User> userByUid = listToMap(userService.massGetUser(uids), User::getUid);
            return dataFetcher.constructRepresentativesResponse(representativesByClient, userByUid);
        }
    }

}

