package ru.yandex.direct.api.v5.entity.creatives.converter;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.creatives.BusinessTypeEnum;
import com.yandex.direct.api.v5.creatives.CpcVideoCreativeFieldEnum;
import com.yandex.direct.api.v5.creatives.CpcVideoCreativeGet;
import com.yandex.direct.api.v5.creatives.CpmVideoCreativeFieldEnum;
import com.yandex.direct.api.v5.creatives.CpmVideoCreativeGet;
import com.yandex.direct.api.v5.creatives.CreativeFieldEnum;
import com.yandex.direct.api.v5.creatives.CreativeGetItem;
import com.yandex.direct.api.v5.creatives.CreativeTypeEnum;
import com.yandex.direct.api.v5.creatives.ObjectFactory;
import com.yandex.direct.api.v5.creatives.SmartCreativeFieldEnum;
import com.yandex.direct.api.v5.creatives.SmartCreativeGet;
import com.yandex.direct.api.v5.creatives.VideoCreativeGetBase;
import com.yandex.direct.api.v5.creatives.VideoExtensionCreativeFieldEnum;
import com.yandex.direct.api.v5.creatives.VideoExtensionCreativeGet;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.entity.creatives.container.CreativeGetContainer;
import ru.yandex.direct.api.v5.entity.creatives.delegate.CreativeAnyFieldEnum;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;

import static com.yandex.direct.api.v5.creatives.CreativeGetItem.PropInfo.CPC_VIDEO_CREATIVE;
import static com.yandex.direct.api.v5.creatives.CreativeGetItem.PropInfo.CPM_VIDEO_CREATIVE;
import static com.yandex.direct.api.v5.creatives.CreativeGetItem.PropInfo.SMART_CREATIVE;
import static com.yandex.direct.api.v5.creatives.CreativeGetItem.PropInfo.VIDEO_EXTENSION_CREATIVE;
import static java.util.stream.Collectors.toList;
import static one.util.streamex.MoreCollectors.toEnumSet;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component(GetResponseConverter.COMPONENT_NAME)
@ParametersAreNonnullByDefault
public class GetResponseConverter {

    static final String COMPONENT_NAME = "creativesResponseConverter";
    private static final ObjectFactory FACTORY = new ObjectFactory();
    private static final Map<CreativeBusinessType, BusinessTypeEnum> BUSINESS_TYPE_API_BY_CORE =
            getBusinessTypeApiByCore();

    private final EnumPropertyFilter<CreativeFieldEnum> baseCreativePropertyFilter;
    private final EnumPropertyFilter<VideoExtensionCreativeFieldEnum> videoAdditionCreativePropertyFilter;
    private final EnumPropertyFilter<CpcVideoCreativeFieldEnum> cpcVideoCreativePropertyFilter;
    private final EnumPropertyFilter<CpmVideoCreativeFieldEnum> cpmVideoCreativePropertyFilter;
    private final EnumPropertyFilter<SmartCreativeFieldEnum> smartCreativePropertyFilter;

    @Autowired
    public GetResponseConverter(PropertyFilter propertyFilter) {
        this.baseCreativePropertyFilter = EnumPropertyFilter.from(CreativeFieldEnum.class, propertyFilter);
        this.videoAdditionCreativePropertyFilter =
                EnumPropertyFilter.from(VideoExtensionCreativeFieldEnum.class, propertyFilter);
        this.cpcVideoCreativePropertyFilter = EnumPropertyFilter.from(CpcVideoCreativeFieldEnum.class, propertyFilter);
        this.cpmVideoCreativePropertyFilter = EnumPropertyFilter.from(CpmVideoCreativeFieldEnum.class, propertyFilter);
        this.smartCreativePropertyFilter = EnumPropertyFilter.from(SmartCreativeFieldEnum.class, propertyFilter);
    }

    // converters

    public static CreativeGetItem convertToApiItem(CreativeGetContainer container) {
        CreativeGetItem creativeGetItem = FACTORY.createCreativeGetItem();

        Creative creative = container.getCreative();

        creativeGetItem
                .withId(creative.getId())
                .withType(convertType(creative.getType()))
                .withName(creative.getName())
                .withPreviewUrl(creative.getPreviewUrl())
                .withWidth(ifNotNull(creative.getWidth(), Long::intValue))
                .withHeight(ifNotNull(creative.getHeight(), Long::intValue))
                .withThumbnailUrl(creative.getLivePreviewUrl())
                .withAssociated(container.isUsedInAds() ? YesNoEnum.YES : YesNoEnum.NO)
                .withIsAdaptive(creative.getIsAdaptive() ? YesNoEnum.YES : YesNoEnum.NO);

        if (creative.getType() == CreativeType.VIDEO_ADDITION_CREATIVE) {
            creativeGetItem.withVideoExtensionCreative(convertVideoAdditionCreative(creative));
        }
        if (creative.getType() == CreativeType.CPC_VIDEO_CREATIVE) {
            creativeGetItem.withCpcVideoCreative(convertCpcVideoCreative(creative));
        }
        if (creative.getType() == CreativeType.CPM_VIDEO_CREATIVE) {
            creativeGetItem.withCpmVideoCreative(convertCpmVideoCreative(creative));
        }

        if (creative.getType() == CreativeType.PERFORMANCE) {
            creativeGetItem.withSmartCreative(convertSmartCreative(creative));
        }

        return creativeGetItem;
    }

    private static <T extends VideoCreativeGetBase> void initVideoBase(T creativeGet, Creative creative) {
        if (creative.getDuration() != null) {
            creativeGet.withDuration(creative.getDuration().intValue());
        }
    }

    private static VideoExtensionCreativeGet convertVideoAdditionCreative(Creative creative) {
        VideoExtensionCreativeGet videoExtensionCreativeGet = FACTORY.createVideoExtensionCreativeGet();
        initVideoBase(videoExtensionCreativeGet, creative);
        return videoExtensionCreativeGet;
    }

    private static CpcVideoCreativeGet convertCpcVideoCreative(Creative creative) {
        CpcVideoCreativeGet cpcVideoCreativeGet = FACTORY.createCpcVideoCreativeGet();
        initVideoBase(cpcVideoCreativeGet, creative);
        return cpcVideoCreativeGet;
    }

    private static CpmVideoCreativeGet convertCpmVideoCreative(Creative creative) {
        CpmVideoCreativeGet cpmVideoCreativeGet = FACTORY.createCpmVideoCreativeGet();
        initVideoBase(cpmVideoCreativeGet, creative);
        return cpmVideoCreativeGet;
    }

    private static Map<CreativeBusinessType, BusinessTypeEnum> getBusinessTypeApiByCore() {
        return Map.of(CreativeBusinessType.AUTO, BusinessTypeEnum.AUTOMOBILES,
                CreativeBusinessType.FLIGHTS, BusinessTypeEnum.FLIGHTS,
                CreativeBusinessType.HOTELS, BusinessTypeEnum.HOTELS,
                CreativeBusinessType.OTHER, BusinessTypeEnum.OTHER,
                CreativeBusinessType.REALTY, BusinessTypeEnum.REALTY,
                CreativeBusinessType.RETAIL, BusinessTypeEnum.RETAIL);
    }

    private static SmartCreativeGet convertSmartCreative(Creative creative) {
        return FACTORY.createSmartCreativeGet()
                .withBusinessType(BUSINESS_TYPE_API_BY_CORE.get(creative.getBusinessType()))
                .withCreativeGroupId(creative.getCreativeGroupId())
                .withCreativeGroupName(creative.getGroupName());
    }

    private static CreativeTypeEnum convertType(CreativeType type) {
        switch (type) {
            case CANVAS:
                return CreativeTypeEnum.IMAGE_CREATIVE;
            case HTML5_CREATIVE:
                return CreativeTypeEnum.HTML_5_CREATIVE;
            case VIDEO_ADDITION_CREATIVE:
                return CreativeTypeEnum.VIDEO_EXTENSION_CREATIVE;
            case CPC_VIDEO_CREATIVE:
                return CreativeTypeEnum.CPC_VIDEO_CREATIVE;
            case CPM_VIDEO_CREATIVE:
                return CreativeTypeEnum.CPM_VIDEO_CREATIVE;
            case PERFORMANCE:
                return CreativeTypeEnum.SMART_CREATIVE;
            default:
                throw new IllegalStateException("No such value: " + type);
        }
    }


    // filters

    public void filterProperties(List<CreativeGetItem> getItems, Set<CreativeAnyFieldEnum> requestedFields) {
        List<String> propertyNames = new ArrayList<>();

        Map<? extends Class<?>, List<CreativeAnyFieldEnum>> requestedFieldsByType =
                requestedFields.stream().collect(Collectors.groupingBy(CreativeAnyFieldEnum::getEnumClass));

        filterProperty(VIDEO_EXTENSION_CREATIVE.propertyName, VideoExtensionCreativeFieldEnum.class,
                CreativeGetItem::getVideoExtensionCreative, videoAdditionCreativePropertyFilter,
                getItems, requestedFieldsByType, propertyNames);

        filterProperty(CPC_VIDEO_CREATIVE.propertyName, CpcVideoCreativeFieldEnum.class,
                CreativeGetItem::getCpcVideoCreative, cpcVideoCreativePropertyFilter,
                getItems, requestedFieldsByType, propertyNames);

        filterProperty(CPM_VIDEO_CREATIVE.propertyName, CpmVideoCreativeFieldEnum.class,
                CreativeGetItem::getCpmVideoCreative, cpmVideoCreativePropertyFilter,
                getItems, requestedFieldsByType, propertyNames);

        filterProperty(SMART_CREATIVE.propertyName, SmartCreativeFieldEnum.class,
                CreativeGetItem::getSmartCreative, smartCreativePropertyFilter,
                getItems, requestedFieldsByType, propertyNames);

        EnumSet<CreativeFieldEnum> baseAdFields = EnumSet.noneOf(CreativeFieldEnum.class);
        if (requestedFieldsByType.containsKey(CreativeFieldEnum.class)) {
            baseAdFields = getRequestedFieldsByType(requestedFieldsByType.get(CreativeFieldEnum.class),
                    CreativeFieldEnum.class);
        }

        propertyNames.addAll(mapList(baseAdFields, baseCreativePropertyFilter.getEnumToFieldMap()::get));
        baseCreativePropertyFilter.filterPropertiesByNames(getItems, propertyNames);
    }

    private <GetItem, FieldEnum extends Enum<FieldEnum>> void filterProperty(
            String propertyName,
            Class<FieldEnum> clazz,
            Function<CreativeGetItem, GetItem> getTypedCreative,
            EnumPropertyFilter<FieldEnum> propertyFilter,
            List<CreativeGetItem> getItems,
            Map<? extends Class<?>, List<CreativeAnyFieldEnum>> requestedFieldsByType,
            List<String> propertyNames) {
        List<GetItem> typedCreativeItems = getItemsOfType(getItems, getTypedCreative);
        if (!typedCreativeItems.isEmpty()) {
            EnumSet<FieldEnum> creativeFields = EnumSet.noneOf(clazz);

            if (requestedFieldsByType.containsKey(clazz)) {
                creativeFields = getRequestedFieldsByType(requestedFieldsByType.get(clazz), clazz);
                propertyNames.add(propertyName);
            }

            propertyFilter.filterProperties(typedCreativeItems, creativeFields);
        }
    }

    private <T, R> List<R> getItemsOfType(List<T> items, Function<T, R> getter) {
        return items.stream().map(getter).filter(Objects::nonNull).collect(toList());
    }

    private <T extends Enum<T>> EnumSet<T> getRequestedFieldsByType(List<CreativeAnyFieldEnum> adAnyFieldEnums,
                                                                    Class<T> clazz) {
        return StreamEx.of(adAnyFieldEnums)
                .map(e -> clazz.cast(e.getValue()))
                .collect(toEnumSet(clazz));
    }
}

