package ru.yandex.direct.api.v5.entity.creatives.delegate;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.yandex.direct.api.v5.creatives.CreativeGetItem;
import com.yandex.direct.api.v5.creatives.CreativeTypeEnum;
import com.yandex.direct.api.v5.creatives.CreativesSelectionCriteria;
import com.yandex.direct.api.v5.creatives.GetRequest;
import com.yandex.direct.api.v5.creatives.GetResponse;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.creatives.container.CreativeGetContainer;
import ru.yandex.direct.api.v5.entity.creatives.converter.GetResponseConverter;
import ru.yandex.direct.api.v5.entity.creatives.validation.GetCreativesValidationService;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.client.service.checker.ClientAccessCheckerTypeSupportFacade;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.api.v5.entity.creatives.converter.GetRequestConverter.EXTERNAL_TO_INTERNAL_TYPE;
import static ru.yandex.direct.api.v5.entity.creatives.converter.GetRequestConverter.convertTypes;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class GetCreativesDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, CreativeAnyFieldEnum,
        CreativesSelectionCriteria, CreativeGetContainer> {
    private static final Logger logger = LoggerFactory.getLogger(GetCreativesDelegate.class);

    private final GetCreativesValidationService validationService;
    private final GetResponseConverter getResponseConverter;

    private final CreativeRepository creativeRepository;
    private final ShardHelper shardHelper;
    private final ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade;

    @Autowired
    public GetCreativesDelegate(
            ApiAuthenticationSource auth,
            GetCreativesValidationService validationService,
            GetResponseConverter getResponseConverter,
            CreativeRepository creativeRepository, ShardHelper shardHelper,
            ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade) {
        super(ApiPathConverter.forCreatives(), auth);

        this.validationService = validationService;
        this.getResponseConverter = getResponseConverter;
        this.creativeRepository = creativeRepository;
        this.shardHelper = shardHelper;
        this.clientAccessCheckerTypeSupportFacade = clientAccessCheckerTypeSupportFacade;
    }

    @Nullable
    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        return validationService.validateRequest(externalRequest);
    }

    @Override
    public Set<CreativeAnyFieldEnum> extractFieldNames(GetRequest externalRequest) {

        return StreamEx.of(
                externalRequest.getFieldNames().stream().map(CreativeAnyFieldEnum::fromCreativeFieldEnum),
                externalRequest.getVideoExtensionCreativeFieldNames().stream()
                        .map(CreativeAnyFieldEnum::fromVideoExtensionCreativeFieldEnum),
                externalRequest.getCpcVideoCreativeFieldNames().stream()
                        .map(CreativeAnyFieldEnum::fromCpcVideoCreativeFieldEnum),
                externalRequest.getCpmVideoCreativeFieldNames().stream()
                        .map(CreativeAnyFieldEnum::fromCpmVideoCreativeFieldEnum),
                externalRequest.getSmartCreativeFieldNames().stream()
                        .map(CreativeAnyFieldEnum::fromSmartCreativeFieldEnum)
        ).flatMap(identity()).toSet();
    }

    @Override
    public CreativesSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        return externalRequest.getSelectionCriteria();
    }

    @Override
    public List<CreativeGetContainer> get(GenericGetRequest<CreativeAnyFieldEnum, CreativesSelectionCriteria> request) {
        CreativesSelectionCriteria selectionCriteria = request.getSelectionCriteria();

        ClientId clientId = auth.getChiefSubclient().getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Long> ids = selectionCriteria.getIds();

        List<CreativeTypeEnum> creativeTypeEnums = selectionCriteria.getTypes();
        Set<CreativeType> types = CollectionUtils.isEmpty(creativeTypeEnums)
                ? ImmutableSet.copyOf(EXTERNAL_TO_INTERNAL_TYPE.values())
                : convertTypes(creativeTypeEnums);

        LimitOffset limitOffset = request.getLimitOffset();

        Set<Long> existingCreativeIds;
        if (ids.isEmpty()) {
            existingCreativeIds = creativeRepository.getExistingClientCreativeIds(shard, clientId, ids, types,
                    limitOffset);
        } else {
            Set<Long> verifiedCreativeIds =
                    clientAccessCheckerTypeSupportFacade.sendToCheck(Map.of(Creative.class, ids), clientId)
                            .getOrDefault(Creative.class, Collections.emptySet());

            if (verifiedCreativeIds.isEmpty()) {
                existingCreativeIds = verifiedCreativeIds;
            } else {
                existingCreativeIds = creativeRepository.getExistingClientCreativeIds(shard, clientId,
                        verifiedCreativeIds, types, limitOffset);
            }
        }

        List<Creative> creatives = creativeRepository.getCreatives(shard, clientId, existingCreativeIds);
        Set<Long> usedCreativeIds = creativeRepository.getUsedCreativeIds(shard, clientId, existingCreativeIds);

        return StreamEx.of(creatives)
                .sorted((Comparator.comparing(Creative::getId)))
                .map(creative -> new CreativeGetContainer()
                        .withCreative(creative)
                        .withUsedInAds(usedCreativeIds.contains(creative.getId())))
                .toList();
    }

    @Override
    public GetResponse convertGetResponse(List<CreativeGetContainer> result, Set<CreativeAnyFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        logger.debug("convert response - result {}, requestedFields {}, limitedBy {}", result, requestedFields,
                limitedBy);

        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        if (!result.isEmpty()) {
            List<CreativeGetItem> getItems = mapList(result, GetResponseConverter::convertToApiItem);
            getResponseConverter.filterProperties(getItems, requestedFields);
            response.withCreatives(getItems);
        }

        return response;
    }

}
