package ru.yandex.direct.api.v5.entity.creatives.validation;

import java.util.HashSet;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.creatives.CreativeTypeEnum;
import com.yandex.direct.api.v5.creatives.CreativesSelectionCriteria;
import com.yandex.direct.api.v5.creatives.GetRequest;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.creatives.Constants.MAX_ELEMENTS_PER_GET;
import static ru.yandex.direct.api.v5.entity.creatives.converter.GetRequestConverter.EXTERNAL_TO_INTERNAL_TYPE;
import static ru.yandex.direct.api.v5.validation.DefectTypeCreators.wrongSelectionCriteriaUnsupportedFieldValues;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.inSet;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class GetCreativesValidationService {

    static final String SELECTION_CRITERIA_FIELD =
            GetRequest.PropInfo.SELECTION_CRITERIA.schemaName.getLocalPart();
    static final String IDS_FIELD = CreativesSelectionCriteria.PropInfo.IDS.schemaName.getLocalPart();
    static final String TYPES_FIELD = CreativesSelectionCriteria.PropInfo.TYPES.schemaName.getLocalPart();

    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);

        vb.checkBy(GetRequestGeneralValidator::validateRequest);
        vb.item(externalRequest.getSelectionCriteria(), SELECTION_CRITERIA_FIELD)
                .checkBy(sc -> validateSelectionCriteria(sc), When.notNull());

        return vb.getResult();
    }

    private static ValidationResult<CreativesSelectionCriteria, DefectType> validateSelectionCriteria(
            CreativesSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<CreativesSelectionCriteria, DefectType> vb =
                ItemValidationBuilder.of(selectionCriteria);

        vb.item(selectionCriteria.getIds(), IDS_FIELD)
                .check(eachNotNull())
                .check(maxListSize(MAX_ELEMENTS_PER_GET), maxElementsInSelection(MAX_ELEMENTS_PER_GET));

        Set<CreativeTypeEnum> allCreativeTypes = new HashSet<>(EXTERNAL_TO_INTERNAL_TYPE.keySet());

        vb.list(selectionCriteria.getTypes(), TYPES_FIELD)
                .check(eachNotNull())
                .checkEach(inSet(allCreativeTypes),
                        wrongSelectionCriteriaUnsupportedFieldValues(TYPES_FIELD,
                                mapList(allCreativeTypes, Enum::name)));

        return vb.getResult();
    }
}
