package ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;

import com.yandex.direct.api.v5.dynamicfeedadtargets.ConditionsArray;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DynamicFeedAdTargetAddItem;
import com.yandex.direct.api.v5.dynamicfeedadtargets.FeedCondition;
import com.yandex.direct.api.v5.general.AdTargetStateSelectionEnum;
import com.yandex.direct.api.v5.general.StringConditionOperatorEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTargetState;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.utils.converter.Converters;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.core.entity.performancefilter.model.Operator.EQUALS;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class CommonConverter {
    protected static final Map<AdTargetStateSelectionEnum, DynamicTextAdTargetState> API_TO_CORE_STATES =
            Map.of(AdTargetStateSelectionEnum.OFF, DynamicTextAdTargetState.OFF,
                    AdTargetStateSelectionEnum.ON, DynamicTextAdTargetState.ON,
                    AdTargetStateSelectionEnum.SUSPENDED, DynamicTextAdTargetState.SUSPENDED,
                    AdTargetStateSelectionEnum.DELETED, DynamicTextAdTargetState.DELETED);

    protected static final Map<StringConditionOperatorEnum, Operator> API_TO_CORE_FEED_OPERATOR =
            Map.of(StringConditionOperatorEnum.GREATER_THAN, Operator.GREATER,
                    StringConditionOperatorEnum.LESS_THAN, Operator.LESS,
                    StringConditionOperatorEnum.EQUALS_ANY, Operator.EQUALS,
                    StringConditionOperatorEnum.IN_RANGE, Operator.RANGE,
                    StringConditionOperatorEnum.CONTAINS_ANY, Operator.CONTAINS,
                    StringConditionOperatorEnum.NOT_CONTAINS_ALL, Operator.NOT_CONTAINS,
                    StringConditionOperatorEnum.EXISTS, Operator.EXISTS);

    protected static final Map<Operator, StringConditionOperatorEnum> CORE_TO_API_FEED_OPERATOR =
            invert(API_TO_CORE_FEED_OPERATOR);

    private static <T, U> Map<U, T> invert(Map<T, U> source) {
        return EntryStream.of(source)
                .invert()
                .toMap();
    }

    protected static BigDecimal convertPrice(Long price) {
        if (price != null) {
            return convertToDbPrice(price);
        }
        return null;
    }

    protected static DynamicAdTargetTab convertTab(@Nullable List<DynamicFeedRule> conditions) {
        if (conditions == null) {
            return null;
        }
        if (conditions.isEmpty()) {
            return DynamicAdTargetTab.ALL_PRODUCTS;
        }
        if (PerformanceFilterConditionDBFormatParser.isSupportedByTreeTab(conditions)) {
            return DynamicAdTargetTab.TREE;
        }
        return DynamicAdTargetTab.CONDITION;
    }

    private static Operator convertOperator(StringConditionOperatorEnum operator) {
        return Converters.mappingValueConverter(API_TO_CORE_FEED_OPERATOR).convert(operator);
    }

    protected static List<DynamicFeedRule> convertConditionsAdd(FilterSchema filterSchema,
                                                                DynamicFeedAdTargetAddItem addItem) {
        List<FeedCondition> conditionItems = nvl(ifNotNull(addItem.getConditions(), ConditionsArray::getItems),
                List.of());
        return convertConditions(filterSchema, conditionItems, addItem.getAvailableItemsOnly());
    }

    private static List<DynamicFeedRule> convertConditions(FilterSchema filterSchema,
                                                           List<FeedCondition> feedConditions,
                                                           YesNoEnum availableItemsOnly) {
        List<DynamicFeedRule> conditions = mapList(feedConditions,
                condition -> new DynamicFeedRule<>(
                        condition.getOperand(),
                        convertOperator(condition.getOperator()),
                        "[\"" + String.join("\",\"", condition.getArguments()) + "\"]")
        );

        if (Objects.equals(availableItemsOnly, YesNoEnum.YES)) {
            conditions.add(new DynamicFeedRule<>("available", EQUALS, "true"));
        }
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions);

        return conditions;
    }
}
