package ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter;

import java.util.List;
import java.util.Map;

import com.yandex.direct.api.v5.dynamicfeedadtargets.AddRequest;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DynamicFeedAdTargetAddItem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter.CommonConverter.convertConditionsAdd;
import static ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter.CommonConverter.convertPrice;
import static ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter.CommonConverter.convertTab;
import static ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component(DynamicFeedAdTargetsAddRequestConverter.COMPONENT_NAME)
public class DynamicFeedAdTargetsAddRequestConverter {
    public static final String COMPONENT_NAME = "DynamicFeedAdTargetsAddRequestConverter";

    private final FeedService feedService;
    private final PerformanceFilterStorage performanceFilterStorage;

    @Autowired
    public DynamicFeedAdTargetsAddRequestConverter(FeedService feedService,
                                                   PerformanceFilterStorage performanceFilterStorage) {
        this.feedService = feedService;
        this.performanceFilterStorage = performanceFilterStorage;
    }

    public List<DynamicFeedAdTarget> convert(ClientId clientId, AddRequest addRequest) {
        var dynamicFeedAdTargets = addRequest.getDynamicFeedAdTargets();
        var adGroupIds = mapList(dynamicFeedAdTargets, DynamicFeedAdTargetAddItem::getAdGroupId);
        var feedByAdGroupId = feedService.getFeedByDynamicAdGroupId(clientId, adGroupIds);
        return mapList(dynamicFeedAdTargets,
                dynamicFeedAdTargetAddItem -> convertAddItem(dynamicFeedAdTargetAddItem, feedByAdGroupId));
    }

    protected DynamicFeedAdTarget convertAddItem(DynamicFeedAdTargetAddItem addItem, Map<Long, Feed> feedByAdGroupId) {
        Feed feed = feedByAdGroupId.get(addItem.getAdGroupId());
        BusinessType businessType = feed.getBusinessType();
        FeedType feedType = feed.getFeedType();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType);
        List<DynamicFeedRule> conditions = convertConditionsAdd(filterSchema, addItem);

        return new DynamicFeedAdTarget()
                .withAdGroupId(addItem.getAdGroupId())
                .withConditionName(addItem.getName())
                .withPrice(convertPrice(addItem.getBid()))
                .withPriceContext(convertPrice(addItem.getContextBid()))
                .withIsSuspended(false)
                .withBusinessType(businessType)
                .withFeedType(feedType)
                .withCondition(conditions)
                .withConditionHash(getHashForDynamicFeedRules(conditions))
                .withTab(convertTab(conditions));
    }
}
