package ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.dynamicfeedadtargets.ConditionsArray;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DynamicFeedAdTargetFieldEnum;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DynamicFeedAdTargetGetItem;
import com.yandex.direct.api.v5.dynamicfeedadtargets.FeedCondition;
import com.yandex.direct.api.v5.dynamicfeedadtargets.GetResponse;
import com.yandex.direct.api.v5.dynamicfeedadtargets.ObjectFactory;
import com.yandex.direct.api.v5.general.ConditionTypeEnum;
import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.GeneralUtil;
import ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.container.DynamicFeedAdTargetGetContainer;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.converter.CommonConverter.CORE_TO_API_FEED_OPERATOR;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component(GetResponseConverter.COMPONENT_NAME)
public class GetResponseConverter {
    public static final String COMPONENT_NAME = "dynamicfeedadtargetsResponseConverter";

    private static final Predicate<DynamicFeedRule> IS_AVAILABLE_CONDITION =
            condition -> Objects.equals(condition.getFieldName(), FilterSchema.AVAILABLE);
    private static final Predicate<DynamicFeedRule> IS_NOT_AVAILABLE_CONDITION =
            Predicate.not(IS_AVAILABLE_CONDITION);

    private static final ObjectFactory FACTORY = new ObjectFactory();

    private final EnumPropertyFilter<DynamicFeedAdTargetFieldEnum> propertyFilter;
    private final CampaignService campaignService;

    public GetResponseConverter(PropertyFilter propertyFilter,
                                CampaignService campaignService) {
        this.propertyFilter = EnumPropertyFilter.from(DynamicFeedAdTargetFieldEnum.class, propertyFilter);
        this.campaignService = campaignService;
    }

    public GetResponse getGetResponse(ClientId clientId,
                                      List<DynamicFeedAdTarget> dynamicFeedAdTargets,
                                      Set<DynamicFeedAdTargetFieldEnum> requestedFields,
                                      @Nullable Long limitedBy) {
        var containers = getDynamicFeedAdTargetGetContainers(clientId, dynamicFeedAdTargets);
        List<DynamicFeedAdTargetGetItem> getItems = mapList(containers, this::convert);
        propertyFilter.filterProperties(getItems, requestedFields);

        GetResponse response = new GetResponse()
                .withDynamicFeedAdTargets(getItems)
                .withLimitedBy(limitedBy);

        return response;
    }

    private List<DynamicFeedAdTargetGetContainer> getDynamicFeedAdTargetGetContainers(ClientId clientId,
                                                                                      List<DynamicFeedAdTarget> dynamicFeedAdTargets) {
        List<Long> campaignIds = mapList(dynamicFeedAdTargets, DynamicFeedAdTarget::getCampaignId);
        List<Campaign> campaigns = campaignService.getCampaigns(clientId, campaignIds);
        Map<Long, Campaign> campaignById = listToMap(campaigns, Campaign::getId);

        return mapList(dynamicFeedAdTargets, condition ->
                new DynamicFeedAdTargetGetContainer(condition, campaignById.get(condition.getCampaignId())));
    }

    public DynamicFeedAdTargetGetItem convert(DynamicFeedAdTargetGetContainer dynamicFeedAdTargetGetContainer) {
        var dynamicFeedAdTarget = dynamicFeedAdTargetGetContainer.getDynamicFeedAdTarget();
        Long campaignId = dynamicFeedAdTargetGetContainer.getCampaign().getId();
        Currency currency = dynamicFeedAdTargetGetContainer.getCampaign().getCurrency().getCurrency();
        StateEnum state = convertState(dynamicFeedAdTarget);

        List<FeedCondition> conditionItems = convertConditions(dynamicFeedAdTarget.getCondition());
        ConditionsArray conditionsArray = Optional.of(conditionItems)
                .filter(CollectionUtils::isNotEmpty)
                .map(ci -> new ConditionsArray().withItems(ci))
                .orElse(null);
        JAXBElement<ConditionsArray> conditions =
                FACTORY.createDynamicFeedAdTargetGetItemConditions(conditionsArray);
        YesNoEnum availableItemsOnly = getAvailableItemsOnly(dynamicFeedAdTarget.getCondition());

        return new DynamicFeedAdTargetGetItem()
                .withId(dynamicFeedAdTarget.getDynamicConditionId())
                .withAdGroupId(dynamicFeedAdTarget.getAdGroupId())
                .withCampaignId(campaignId)
                .withName(dynamicFeedAdTarget.getConditionName())
                .withBid(convertPrice(dynamicFeedAdTarget.getPrice(), currency))
                .withContextBid(convertPrice(dynamicFeedAdTarget.getPriceContext(), currency))
                .withState(state)
                .withConditions(conditions)
                .withConditionType(convertConditionType(dynamicFeedAdTarget.getTab()))
                .withAvailableItemsOnly(availableItemsOnly);
    }

    protected Long convertPrice(BigDecimal price, Currency currency) {
        BigDecimal priceToConvert = price;
        if (price == null || price.compareTo(BigDecimal.ZERO) == 0) {
            priceToConvert = currency.getMinPrice();
        }
        return priceToConvert.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    protected StateEnum convertState(DynamicFeedAdTarget dynamicFeedAdTarget) {
        if (dynamicFeedAdTarget.getId() == null) {
            return StateEnum.DELETED;
        }

        if (dynamicFeedAdTarget.getIsSuspended()) {
            return StateEnum.SUSPENDED;
        }

        // StateEnum.OFF не отдаем, см DIRECT-155960: сломался статус ON у DynamicAdTextTarget в апи
        // Если он правда нужен, нужно прикручивать статус с группы или придумывать что-то с аггр. статусами
        return StateEnum.ON;
    }

    protected ConditionTypeEnum convertConditionType(DynamicAdTargetTab tab) {
        return tab == DynamicAdTargetTab.ALL_PRODUCTS
                ? ConditionTypeEnum.ITEMS_ALL
                : ConditionTypeEnum.ITEMS_SUBSET;
    }

    protected List<FeedCondition> convertConditions(List<DynamicFeedRule> conditions) {
        if (conditions == null) {
            return List.of();
        }
        return StreamEx.of(conditions)
                .filter(IS_NOT_AVAILABLE_CONDITION)
                .map(this::convertCondition)
                .toList();
    }

    private FeedCondition convertCondition(DynamicFeedRule rule) {
        return new FeedCondition()
                .withOperand(rule.getFieldName())
                .withOperator(CORE_TO_API_FEED_OPERATOR.get(rule.getOperator()))
                .withArguments(getArguments(rule.getStringValue()));
    }

    private String[] getArguments(String stringValue) {
        if (stringValue.length() > 2 && stringValue.startsWith("[\"")) {
            stringValue = stringValue.substring(2, stringValue.length() - 2);
        }
        return stringValue.split("\",\"");
    }

    private YesNoEnum getAvailableItemsOnly(List<DynamicFeedRule> conditions) {
        boolean hasAvailableCondition = conditions != null && conditions.stream().anyMatch(IS_AVAILABLE_CONDITION);
        return GeneralUtil.yesNoFromBool(hasAvailableCondition);
    }
}
