package ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.validation;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.dynamicfeedadtargets.AddRequest;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DeleteRequest;
import com.yandex.direct.api.v5.dynamicfeedadtargets.DynamicFeedAdTargetAddItem;
import com.yandex.direct.api.v5.dynamicfeedadtargets.GetRequest;
import com.yandex.direct.api.v5.dynamicfeedadtargets.ResumeRequest;
import com.yandex.direct.api.v5.dynamicfeedadtargets.SuspendRequest;
import com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria;
import com.yandex.direct.api.v5.general.IdsCriteria;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.AD_GROUP_IDS;
import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.CAMPAIGN_IDS;
import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.IDS;
import static ru.yandex.direct.api.v5.entity.dynamicfeedadtargets.DynamicFeedAdTargetsDefectTypes.adGroupWithoutFeed;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.inSet;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class DynamicFeedAdTargetsValidationService {
    private static final int MAX_ELEMENTS_PER_REQUEST = 1_000;
    private static final int MAX_CAMPAIGN_IDS_PER_REQUEST = 2;
    private static final int MAX_AD_GROUP_IDS_PER_REQUEST = 1_000;
    private static final int MAX_FILTER_IDS_PER_REQUEST = 10_000;

    public static final String REQUIRED_FIELDS_ERROR = String.join(", ", List.of(
            IDS.schemaName.getLocalPart(),
            AD_GROUP_IDS.schemaName.getLocalPart(),
            CAMPAIGN_IDS.schemaName.getLocalPart()));

    private final FeedService feedService;

    @Autowired
    public DynamicFeedAdTargetsValidationService(FeedService feedService) {
        this.feedService = feedService;
    }

    public ValidationResult<AddRequest, DefectType> validateAddRequest(ClientId clientId, AddRequest addRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(addRequest);
        ListValidationBuilder<DynamicFeedAdTargetAddItem, DefectType> lvb =
                vb.list(addRequest.getDynamicFeedAdTargets(), AddRequest.PropInfo.DYNAMIC_FEED_AD_TARGETS.propertyName);
        lvb.checkEach(notNull())
                .check(maxListSize(MAX_ELEMENTS_PER_REQUEST), maxElementsPerRequest(MAX_ELEMENTS_PER_REQUEST));
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        // Валидируем наличие фида (и попутно тип группы, ибо для нединамической группы фид не вернется)
        List<Long> adGroupIds = mapList(addRequest.getDynamicFeedAdTargets(), DynamicFeedAdTargetAddItem::getAdGroupId);
        Map<Long, Feed> feedByAdGroupId = feedService.getFeedByDynamicAdGroupId(clientId, adGroupIds);
        lvb.checkEachBy(tg -> validateAddItem(tg, feedByAdGroupId), When.isValid());
        return vb.getResult();
    }

    private ValidationResult<DynamicFeedAdTargetAddItem, DefectType> validateAddItem(DynamicFeedAdTargetAddItem addItem,
                                                                                     Map<Long, Feed> feedByAdGroupId) {
        ItemValidationBuilder<DynamicFeedAdTargetAddItem, DefectType> vb = ItemValidationBuilder.of(addItem);
        vb.item(addItem.getAdGroupId(), "AdGroupId")
                .check(inSet(feedByAdGroupId.keySet()), adGroupWithoutFeed());
        return vb.getResult();
    }

    public ValidationResult<GetRequest, DefectType> validateGetRequest(GetRequest getRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(getRequest);
        AdTargetsSelectionCriteria selectionCriteria = getRequest.getSelectionCriteria();

        ItemValidationBuilder<AdTargetsSelectionCriteria, DefectType> vbCriteria =
                vb.item(selectionCriteria, "SelectionCriteria");
        vbCriteria
                .check(notNull())
                .check(this::atLeastOneIdFieldIsPresent, When.isValid());
        validateIds(vbCriteria, selectionCriteria.getIds(), "Ids", MAX_FILTER_IDS_PER_REQUEST);
        validateIds(vbCriteria, selectionCriteria.getAdGroupIds(), "AdGroupIds", MAX_AD_GROUP_IDS_PER_REQUEST);
        validateIds(vbCriteria, selectionCriteria.getCampaignIds(), "CampaignIds", MAX_CAMPAIGN_IDS_PER_REQUEST);

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes, When.notNull());
        return vb.getResult();
    }

    private DefectType atLeastOneIdFieldIsPresent(AdTargetsSelectionCriteria criteria) {
        if (!criteria.getIds().isEmpty()
                || !criteria.getAdGroupIds().isEmpty()
                || !criteria.getCampaignIds().isEmpty()) {
            return null;
        }
        return missedParamsInSelection(REQUIRED_FIELDS_ERROR);
    }

    public ValidationResult<DeleteRequest, DefectType> validateDeleteRequest(DeleteRequest deleteRequest) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(deleteRequest);
        validateSelectionCriteria(vb, deleteRequest.getSelectionCriteria());
        return vb.getResult();
    }

    public ValidationResult<SuspendRequest, DefectType> validateSuspendRequest(SuspendRequest suspendRequest) {
        ItemValidationBuilder<SuspendRequest, DefectType> vb = ItemValidationBuilder.of(suspendRequest);
        validateSelectionCriteria(vb, suspendRequest.getSelectionCriteria());
        return vb.getResult();
    }

    public ValidationResult<ResumeRequest, DefectType> validateResumeRequest(ResumeRequest resumeRequest) {
        ItemValidationBuilder<ResumeRequest, DefectType> vb = ItemValidationBuilder.of(resumeRequest);
        validateSelectionCriteria(vb, resumeRequest.getSelectionCriteria());
        return vb.getResult();
    }

    private void validateSelectionCriteria(ItemValidationBuilder<?, DefectType> vb, IdsCriteria selectionCriteria) {
        ItemValidationBuilder<IdsCriteria, DefectType> item = vb.item(selectionCriteria, "SelectionCriteria");
        validateIds(item, selectionCriteria.getIds(), "Ids", MAX_FILTER_IDS_PER_REQUEST);
    }

    private static void validateIds(ItemValidationBuilder<?, DefectType> vb, List<Long> ids, String name, int max) {
        vb.list(ids, name)
                .checkEach(notNull())
                .check(maxListSize(max), maxElementsInSelection(max));
    }
}
