package ru.yandex.direct.api.v5.entity.dynamictextadtargets.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.dynamictextadtargets.AddRequest;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageAddItem;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageCondition;
import com.yandex.direct.api.v5.general.PriorityEnum;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Component(DynamicTextAdTargetsAddRequestConverter.COMPONENT_NAME)
public class DynamicTextAdTargetsAddRequestConverter {

    public static final String COMPONENT_NAME = "DynamicTextAdTargetsAddRequestConverter";

    private final AdGroupService adGroupService;

    @Autowired
    public DynamicTextAdTargetsAddRequestConverter(AdGroupService adGroupService) {
        this.adGroupService = adGroupService;
    }

    public List<DynamicTextAdTarget> convert(ClientId clientId, AddRequest addRequest) {
        Map<Long, DbStrategy> strategiesByAdGroupIds =
                adGroupService.getStrategiesByAdGroupIds(clientId, getReferencedAdGroupIds(addRequest));

        return mapList(addRequest.getWebpages(), webpage -> convertWebpage(webpage, strategiesByAdGroupIds));
    }

    protected DynamicTextAdTarget convertWebpage(WebpageAddItem webpage, Map<Long, DbStrategy> strategiesByAdGroupIds) {
        DbStrategy dbStrategy = strategiesByAdGroupIds.get(webpage.getAdGroupId());

        DynamicTextAdTarget dynamicTextAdTarget = new DynamicTextAdTarget()
                .withConditionName(webpage.getName())
                .withAdGroupId(webpage.getAdGroupId())
                .withCondition(convertCondition(webpage.getConditions()))
                .withPrice(convertPrice(webpage.getBid()))
                .withPriceContext(convertPrice(webpage.getContextBid()))
                .withAutobudgetPriority(convertPriority(webpage.getStrategyPriority(), dbStrategy))
                .withIsSuspended(false);

        dynamicTextAdTarget
                .withConditionHash(DynamicTextAdTargetHashUtils.getHash(dynamicTextAdTarget.getCondition()))
                .withConditionUniqHash(DynamicTextAdTargetHashUtils.getUniqHash(dynamicTextAdTarget.getCondition()));


        return dynamicTextAdTarget;
    }

    protected List<WebpageRule> convertCondition(List<WebpageCondition> conditions) {
        return mapList(conditions, AddRequestWebpageRuleConverter::convertWebpageCondition);
    }

    protected BigDecimal convertPrice(Long price) {
        if (price != null) {
            return convertToDbPrice(price);
        }
        return null;
    }

    protected Integer convertPriority(PriorityEnum priority, DbStrategy dbStrategy) {
        if (priority != null) {
            return convertStrategyPriority(priority);
        }
        if (dbStrategy != null && dbStrategy.isAutoBudget()) {
            return convertStrategyPriority(PriorityEnum.NORMAL);
        }
        return null;
    }

    private Set<Long> getReferencedAdGroupIds(AddRequest request) {
        return request.getWebpages().stream()
                .map(WebpageAddItem::getAdGroupId)
                .filter(Objects::nonNull)
                .filter(id -> id > 0)
                .collect(Collectors.toSet());
    }
}
