package ru.yandex.direct.api.v5.entity.dynamictextadtargets.converter;

import java.math.BigDecimal;
import java.util.List;

import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.dynamictextadtargets.ObjectFactory;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageCondition;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageGetItem;
import com.yandex.direct.api.v5.dynamictextadtargets.WebpageTypeEnum;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.general.StateEnum;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ConverterUtils;
import ru.yandex.direct.api.v5.entity.dynamictextadtargets.StatusClarificationTranslations;
import ru.yandex.direct.api.v5.entity.dynamictextadtargets.container.DynamicTextAdTargetGetContainer;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;

import static com.yandex.direct.api.v5.general.StateEnum.ON;
import static ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetMapping.isAllPageCondition;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component(GetResponseConverter.COMPONENT_NAME)
public class GetResponseConverter {

    public static final String COMPONENT_NAME = "dynamictextadtargetsResponseConverter";

    private static final ObjectFactory FACTORY = new ObjectFactory();

    private static final StatusClarificationTranslations translations = StatusClarificationTranslations.INSTANCE;

    private final TranslationService translationService;

    public GetResponseConverter(TranslationService translationService) {
        this.translationService = translationService;
    }

    public WebpageGetItem convert(DynamicTextAdTargetGetContainer dynamicTextAdTargetGetContainer) {
        DynamicTextAdTarget dynamicTextAdTarget = dynamicTextAdTargetGetContainer.getDynamicTextAdTarget();
        Long campaignId = dynamicTextAdTargetGetContainer.getCampaign().getId();
        Currency currency = dynamicTextAdTargetGetContainer.getCampaign().getCurrency().getCurrency();
        StateEnum state = convertState(dynamicTextAdTarget);
        return new WebpageGetItem()
                .withId(dynamicTextAdTarget.getDynamicConditionId())
                .withAdGroupId(dynamicTextAdTarget.getAdGroupId())
                .withCampaignId(campaignId)
                .withName(dynamicTextAdTarget.getConditionName())
                .withBid(convertPrice(dynamicTextAdTarget.getPrice(), currency))
                .withContextBid(convertPrice(dynamicTextAdTarget.getPriceContext(), currency))
                .withStrategyPriority(convertStrategyPriority(dynamicTextAdTarget.getAutobudgetPriority()))
                .withState(state)
                .withStatusClarification(getStatusClarification(state))
                .withConditions(convertCondition(dynamicTextAdTarget.getCondition()))
                .withConditionType(convertConditionType(dynamicTextAdTarget.getCondition()));
    }

    protected Long convertPrice(BigDecimal price, Currency currency) {
        BigDecimal priceToConvert = price;
        if (price == null || price.compareTo(BigDecimal.ZERO) == 0) {
            priceToConvert = currency.getMinPrice();
        }
        return priceToConvert.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    protected JAXBElement<PriorityEnum> convertStrategyPriority(Integer priority) {
        return FACTORY.createWebpageGetItemStrategyPriority(ConverterUtils.convertStrategyPriority(priority));
    }

    protected WebpageTypeEnum convertConditionType(List<WebpageRule> conditions) {
        return isAllPageCondition(conditions) ? WebpageTypeEnum.PAGES_ALL : WebpageTypeEnum.PAGES_SUBSET;
    }

    protected List<WebpageCondition> convertCondition(List<WebpageRule> conditions) {
        if (isAllPageCondition(conditions)) {
            return null;
        }
        return mapList(conditions, GetResponseWebpageRuleConverter::convertWebpageRule);
    }

    protected StateEnum convertState(DynamicTextAdTarget dynamicTextAdTarget) {
        if (dynamicTextAdTarget.getId() == null) {
            return StateEnum.DELETED;
        }

        if (dynamicTextAdTarget.getIsSuspended()) {
            return StateEnum.SUSPENDED;
        }

        // StateEnum.OFF не отдаем, см DIRECT-155960: сломался статус ON у DynamicAdTextTarget в апи
        // Если он правда нужен, нужно прикручивать статус с группы или придумывать что-то с аггр. статусами
        return ON;
    }

    protected String getStatusClarification(StateEnum state) {
        switch (state) {
            case ON:
                return translationService.translate(translations.statusOn());
            case OFF:
                return translationService.translate(translations.statusOff());
            case DELETED:
                return translationService.translate(translations.statusDeleted());
            case SUSPENDED:
                return translationService.translate(translations.statusSuspended());
            default:
                return translationService.translate(translations.statusUnknown());
        }
    }
}
