package ru.yandex.direct.api.v5.entity.dynamictextadtargets.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.dynamictextadtargets.GetRequest;
import com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.dynamictextadtargets.GetRequest.PropInfo.SELECTION_CRITERIA;
import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.AD_GROUP_IDS;
import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.CAMPAIGN_IDS;
import static com.yandex.direct.api.v5.general.AdTargetsSelectionCriteria.PropInfo.IDS;
import static java.util.Arrays.asList;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;

@Component
@ParametersAreNonnullByDefault
public class DynamicTextAdTargetsGetRequestValidator {
    public static final int MAX_ADGROUP_IDS_COUNT = 1_000;
    public static final int MAX_CAMPAIGN_IDS_COUNT = 2;
    public static final String REQUIRED_FIELDS_ERROR = String.join(
            ", ", asList(IDS.schemaName.getLocalPart(), AD_GROUP_IDS.schemaName.getLocalPart(),
                    CAMPAIGN_IDS.schemaName.getLocalPart()));

    public ValidationResult<GetRequest, DefectType> validate(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        AdTargetsSelectionCriteria selectionCriteria = externalRequest.getSelectionCriteria();

        vb.item(selectionCriteria, SELECTION_CRITERIA.schemaName.getLocalPart())
                .check(notNull())
                .check(this::atLeastOneIdFieldExisted, When.isValid())
                .checkBy(this::validateIdsSelection, When.isValidAnd(When.notNull()));

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<AdTargetsSelectionCriteria, DefectType> validateIdsSelection(
            AdTargetsSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<AdTargetsSelectionCriteria, DefectType> vb = ItemValidationBuilder.of(selectionCriteria);
        vb.item(selectionCriteria.getIds(), IDS.schemaName.getLocalPart())
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxElementsInSelection(DEFAULT_MAX_IDS_COUNT));
        vb.item(selectionCriteria.getAdGroupIds(), AD_GROUP_IDS.schemaName.getLocalPart())
                .check(maxListSize(MAX_ADGROUP_IDS_COUNT), maxElementsInSelection(MAX_ADGROUP_IDS_COUNT));
        vb.item(selectionCriteria.getCampaignIds(), CAMPAIGN_IDS.schemaName.getLocalPart())
                .check(maxListSize(MAX_CAMPAIGN_IDS_COUNT), maxElementsInSelection(MAX_CAMPAIGN_IDS_COUNT));
        return vb.getResult();
    }

    private DefectType atLeastOneIdFieldExisted(AdTargetsSelectionCriteria sc) {
        if (!sc.getIds().isEmpty() || !sc.getAdGroupIds().isEmpty() || !sc.getCampaignIds().isEmpty()) {
            return null;
        }
        return missedParamsInSelection(REQUIRED_FIELDS_ERROR);
    }
}
