package ru.yandex.direct.api.v5.entity.feeds.converter;

import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.feeds.BusinessTypeEnum;
import com.yandex.direct.api.v5.feeds.FeedFieldEnum;
import com.yandex.direct.api.v5.feeds.FeedGetItem;
import com.yandex.direct.api.v5.feeds.FeedStatusEnum;
import com.yandex.direct.api.v5.feeds.FileFeedFieldEnum;
import com.yandex.direct.api.v5.feeds.FileFeedGet;
import com.yandex.direct.api.v5.feeds.ObjectFactory;
import com.yandex.direct.api.v5.feeds.SourceTypeEnum;
import com.yandex.direct.api.v5.feeds.UrlFeedFieldEnum;
import com.yandex.direct.api.v5.feeds.UrlFeedGet;
import com.yandex.direct.api.v5.general.ArrayOfLong;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.entity.feeds.container.FeedGetContainer;
import ru.yandex.direct.api.v5.entity.feeds.delegate.FeedAnyFieldEnum;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.feed.FeedUtilsKt;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;

import static com.yandex.direct.api.v5.feeds.FeedGetItem.PropInfo.FILE_FEED;
import static com.yandex.direct.api.v5.feeds.FeedGetItem.PropInfo.URL_FEED;
import static java.util.stream.Collectors.toList;
import static one.util.streamex.MoreCollectors.toEnumSet;
import static ru.yandex.direct.api.v5.common.GeneralUtil.yesNoFromBool;
import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.unFakeUrlIfNeeded;
import static ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema.EMPTY_SCHEMA_NAME;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service("feedsGetResponseConverterService")
@ParametersAreNonnullByDefault
public class GetResponseConverterService {
    public static final Map<Source, SourceTypeEnum> SOURCE_API_BY_CORE = Map.of(
            Source.URL, SourceTypeEnum.URL,
            Source.FILE, SourceTypeEnum.FILE
    );
    private static final Map<BusinessType, BusinessTypeEnum> BUSINESS_TYPE_API_BY_CORE = Map.of(
            BusinessType.RETAIL, BusinessTypeEnum.RETAIL,
            BusinessType.HOTELS, BusinessTypeEnum.HOTELS,
            BusinessType.REALTY, BusinessTypeEnum.REALTY,
            BusinessType.AUTO, BusinessTypeEnum.AUTOMOBILES,
            BusinessType.FLIGHTS, BusinessTypeEnum.FLIGHTS,
            BusinessType.OTHER, BusinessTypeEnum.OTHER
    );
    private static final Map<UpdateStatus, FeedStatusEnum> UPDATE_STATUS_API_BY_CORE = Map.of(
            UpdateStatus.NEW, FeedStatusEnum.NEW,
            UpdateStatus.UPDATING, FeedStatusEnum.UPDATING,
            UpdateStatus.OUTDATED, FeedStatusEnum.UPDATING,
            UpdateStatus.DONE, FeedStatusEnum.DONE,
            UpdateStatus.ERROR, FeedStatusEnum.ERROR
    );
    private static final ObjectFactory OBJECT_FACTORY = new ObjectFactory();

    private final PerformanceFilterStorage performanceFilterStorage;
    private final EnumPropertyFilter<FeedFieldEnum> baseFeedPropertyFilter;
    private final EnumPropertyFilter<FileFeedFieldEnum> fileFeedPropertyFilter;
    private final EnumPropertyFilter<UrlFeedFieldEnum> urlFeedPropertyFilter;

    @Autowired
    public GetResponseConverterService(PerformanceFilterStorage performanceFilterStorage,
                                       PropertyFilter propertyFilter) {
        this.performanceFilterStorage = performanceFilterStorage;
        baseFeedPropertyFilter = EnumPropertyFilter.from(FeedFieldEnum.class, propertyFilter);
        fileFeedPropertyFilter = EnumPropertyFilter.from(FileFeedFieldEnum.class, propertyFilter);
        urlFeedPropertyFilter = EnumPropertyFilter.from(UrlFeedFieldEnum.class, propertyFilter);
    }

    public FeedGetItem convertFeed(FeedGetContainer feedContainer) {
        Feed feed = feedContainer.getFeed();
        BusinessTypeEnum businessType = ifNotNull(feed.getBusinessType(), BUSINESS_TYPE_API_BY_CORE::get);
        SourceTypeEnum sourceType = ifNotNull(feed.getSource(), SOURCE_API_BY_CORE::get);
        String filterSchema = getFilterSchema(feed.getBusinessType(), feed.getFeedType(), feed.getSource());
        FeedStatusEnum feedStatus = ifNotNull(feed.getUpdateStatus(), UPDATE_STATUS_API_BY_CORE::get);
        Long offersCount = feed.getOffersCount();
        ArrayOfLong campaignIds = new ArrayOfLong().withItems(feedContainer.getCampaignIds());
        String updatedAt = feed.getLastChange().format(DateTimeFormatter.ISO_LOCAL_DATE_TIME);
        FileFeedGet fileFeed = convertFileFeed(feed.getFilename());
        UrlFeedGet urlFeed = convertUrlFeed(ifNotNull(feed.getUrl(), FeedUtilsKt::unFakeUrlIfNeeded),
                feed.getLogin(),
                feed.getIsRemoveUtm());
        return new FeedGetItem()
                .withId(feed.getId())
                .withName(feed.getName())
                .withBusinessType(businessType)
                .withSourceType(sourceType)
                .withFilterSchema(filterSchema)
                .withStatus(feedStatus)
                .withNumberOfItems(OBJECT_FACTORY.createFeedGetItemNumberOfItems(offersCount))
                .withCampaignIds(OBJECT_FACTORY.createFeedGetItemCampaignIds(campaignIds))
                .withUpdatedAt(OBJECT_FACTORY.createFeedGetItemUpdatedAt(updatedAt))
                .withFileFeed(OBJECT_FACTORY.createFeedGetItemFileFeed(fileFeed))
                .withUrlFeed(OBJECT_FACTORY.createFeedGetItemUrlFeed(urlFeed));
    }

    private String getFilterSchema(@Nullable BusinessType businessType, @Nullable FeedType feedType,
                                   @Nullable Source source) {
        if (businessType == null || (source != Source.SITE && feedType == null)) {
            return null;
        }
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchemaOrNull(businessType, feedType, source);

        return Optional.ofNullable(filterSchema)
                .map(FilterSchema::getFilterSchemaName)
                .orElse(EMPTY_SCHEMA_NAME);
    }


    private static FileFeedGet convertFileFeed(String filename) {
        return new FileFeedGet().withFilename(filename);
    }

    private static UrlFeedGet convertUrlFeed(@Nullable String url, String login, boolean isRemoveUtm) {
        return new UrlFeedGet()
                .withUrl(url)
                .withRemoveUtmTags(yesNoFromBool(isRemoveUtm))
                .withLogin(OBJECT_FACTORY.createUrlFeedGetLogin(login));
    }

    public void filterProperties(List<FeedGetItem> getItems, Set<FeedAnyFieldEnum> requestedFields) {
        Map<? extends Class<?>, List<FeedAnyFieldEnum>> requestedFieldsByType =
                StreamEx.of(requestedFields).groupingBy(FeedAnyFieldEnum::getEnumClass);

        List<String> propertyNames = new ArrayList<>();

        filterProperty(FILE_FEED.propertyName, FileFeedFieldEnum.class,
                it -> ifNotNull(it.getFileFeed(), JAXBElement::getValue),
                fileFeedPropertyFilter, getItems, requestedFieldsByType, propertyNames);

        filterProperty(URL_FEED.propertyName, UrlFeedFieldEnum.class,
                it -> ifNotNull(it.getUrlFeed(), JAXBElement::getValue),
                urlFeedPropertyFilter, getItems, requestedFieldsByType, propertyNames);

        EnumSet<FeedFieldEnum> baseFields = EnumSet.noneOf(FeedFieldEnum.class);
        if (requestedFieldsByType.containsKey(FeedFieldEnum.class)) {
            baseFields = getRequestedFieldsByType(requestedFieldsByType.get(FeedFieldEnum.class), FeedFieldEnum.class);
        }
        propertyNames.addAll(mapList(baseFields, baseFeedPropertyFilter.getEnumToFieldMap()::get));
        baseFeedPropertyFilter.filterPropertiesByNames(getItems, propertyNames);
    }

    private <GetItem, FieldEnum extends Enum<FieldEnum>> void filterProperty(
            String propertyName,
            Class<FieldEnum> clazz,
            Function<FeedGetItem, GetItem> getTypedFeed,
            EnumPropertyFilter<FieldEnum> propertyFilter,
            List<FeedGetItem> getItems,
            Map<? extends Class<?>, List<FeedAnyFieldEnum>> requestedFieldsByType,
            List<String> propertyNames) {
        List<GetItem> typedFeedItems = getItemsOfType(getItems, getTypedFeed);
        if (!typedFeedItems.isEmpty()) {
            EnumSet<FieldEnum> feedFields = EnumSet.noneOf(clazz);

            if (requestedFieldsByType.containsKey(clazz)) {
                feedFields = getRequestedFieldsByType(requestedFieldsByType.get(clazz), clazz);
                propertyNames.add(propertyName);
            }

            propertyFilter.filterProperties(typedFeedItems, feedFields);
        }
    }

    private <T, R> List<R> getItemsOfType(List<T> items, Function<T, R> getter) {
        return items.stream().map(getter).filter(Objects::nonNull).collect(toList());
    }

    private <T extends Enum<T>> EnumSet<T> getRequestedFieldsByType(
            List<FeedAnyFieldEnum> feedAnyFieldEnums,
            Class<T> clazz
    ) {
        return StreamEx.of(feedAnyFieldEnums)
                .map(e -> clazz.cast(e.getValue()))
                .collect(toEnumSet(clazz));
    }
}
