package ru.yandex.direct.api.v5.entity.feeds.validation;


import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.feeds.AddRequest;
import com.yandex.direct.api.v5.feeds.DeleteRequest;
import com.yandex.direct.api.v5.feeds.FeedAddItem;
import com.yandex.direct.api.v5.feeds.FeedUpdateItem;
import com.yandex.direct.api.v5.feeds.FeedsSelectionCriteria;
import com.yandex.direct.api.v5.feeds.SourceTypeEnum;
import com.yandex.direct.api.v5.feeds.UpdateRequest;
import com.yandex.direct.api.v5.feeds.UrlFeedAdd;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.yandex.direct.api.v5.adgroups.GetRequest.PropInfo.SELECTION_CRITERIA;
import static ru.yandex.direct.api.v5.entity.smartadtargets.SmartAdTargetsDefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredButEmpty;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;

@ParametersAreNonnullByDefault
public class FeedsValidationService {

    public static final int MAX_ELEMENTS_PER_REQUEST = 50;
    public static final int MAX_IDS_COUNT = 50;

    public static ValidationResult<FeedsSelectionCriteria, DefectType> validateSelectionCriteria(
            FeedsSelectionCriteria selectionCriteria
    ) {
        ItemValidationBuilder<FeedsSelectionCriteria, DefectType> vb = ItemValidationBuilder.of(selectionCriteria);
        vb.item(selectionCriteria.getIds(), "Ids").check(maxListSize(MAX_IDS_COUNT), maxIdsInSelection());
        return vb.getResult();
    }

    public static ValidationResult<AddRequest, DefectType> validateAdd(AddRequest addRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(addRequest);
        ListValidationBuilder<FeedAddItem, DefectType> lvb =
                vb.list(addRequest.getFeeds(), AddRequest.PropInfo.FEEDS.schemaName.toString());
        lvb.check(maxListSize(MAX_ELEMENTS_PER_REQUEST), maxElementsPerRequest(MAX_ELEMENTS_PER_REQUEST))
                .checkEachBy(validateFile(), When.isValid())
                .checkEachBy(validateUrl(), When.isValid());
        return vb.getResult();
    }

    public static ValidationResult<UpdateRequest, DefectType> validateUpdate(UpdateRequest updateRequest) {
        ItemValidationBuilder<UpdateRequest, DefectType> vb = ItemValidationBuilder.of(updateRequest);
        ListValidationBuilder<FeedUpdateItem, DefectType> lvb =
                vb.list(updateRequest.getFeeds(), AddRequest.PropInfo.FEEDS.schemaName.toString());
        lvb.check(maxListSize(MAX_ELEMENTS_PER_REQUEST), maxElementsPerRequest(MAX_ELEMENTS_PER_REQUEST));
        return vb.getResult();
    }

    public static ValidationResult<DeleteRequest, DefectType> validateDelete(DeleteRequest deleteRequest) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(deleteRequest);
        vb.item(deleteRequest.getSelectionCriteria(), SELECTION_CRITERIA.schemaName.toString())
                .checkBy(FeedsValidationService::validateSelectionCriteria, When.notNull());
        return vb.getResult();
    }

    private static Validator<FeedAddItem, DefectType> validateFile() {
        return item -> {
            ItemValidationBuilder<FeedAddItem, DefectType> vb = ModelItemValidationBuilder.of(item);
            vb.item(item.getFileFeed(), FeedAddItem.PropInfo.FILE_FEED.schemaName.toString())
                    .check(notNull(), requiredButEmpty(), When.isTrue(item.getSourceType() == SourceTypeEnum.FILE));
            return vb.getResult();
        };
    }

    private static Validator<FeedAddItem, DefectType> validateUrl() {
        return item -> {
            ItemValidationBuilder<FeedAddItem, DefectType> vb = ModelItemValidationBuilder.of(item);
            UrlFeedAdd urlFeed = item.getUrlFeed();
            vb.item(urlFeed, FeedAddItem.PropInfo.URL_FEED.schemaName.toString())
                    .check(notNull(), requiredButEmpty(), When.isTrue(item.getSourceType() == SourceTypeEnum.URL));
            return vb.getResult();
        };
    }
}
