package ru.yandex.direct.api.v5.entity.keywordbids.converter;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywordbids.AuctionBidItem;
import com.yandex.direct.api.v5.keywordbids.AuctionBids;
import com.yandex.direct.api.v5.keywordbids.KeywordBidGetItem;
import com.yandex.direct.api.v5.keywordbids.ObjectFactory;
import com.yandex.direct.api.v5.keywordbids.Search;
import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.entity.keywordbids.KeywordBidAnyFieldEnum;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;

import static java.util.Comparator.comparing;
import static ru.yandex.direct.api.v5.entity.bids.converter.get.BsAuctionBidFieldWriter.getShowValue;
import static ru.yandex.direct.api.v5.entity.keywordbids.converter.KeywordBidGetItemWriterComposition.writerCompositionOf;


@ParametersAreNonnullByDefault
public class BsAuctionKeywordBidFieldWriter implements KeywordBidGetItemWriter {

    private static final ObjectFactory FACTORY = new ObjectFactory();
    private static final double TRAFFIC_VOLUME_MULTIPLIER = 10_000.0;

    private final Set<KeywordBidAnyFieldEnum> requiredFields;
    private final Map<Long, KeywordTrafaretData> bsResultsById;

    private final KeywordBidGetItemWriter writer;

    private BsAuctionKeywordBidFieldWriter(Set<KeywordBidAnyFieldEnum> requiredFields,
                                           Map<Long, KeywordTrafaretData> bsResultsById) {
        this.requiredFields = requiredFields;
        this.bsResultsById = bsResultsById;
        writer = writerCompositionOf(getAuctionBidsWriter());
    }

    static BsAuctionKeywordBidFieldWriter createBsAuctionKeywordBidFieldWriter(
            Set<KeywordBidAnyFieldEnum> requiredFields,
            Collection<CompleteBidData<KeywordTrafaretData>> completeBidData) {
        Map<Long, KeywordTrafaretData> bsAuctionDataByBidId =
                StreamEx.of(completeBidData)
                        .mapToEntry(CompleteBidData::getBidId)
                        .invert()
                        .mapValues(CompleteBidData::getDynamicData)
                        .nonNullValues()
                        .mapValues(KeywordBidDynamicData::getBsAuctionData)
                        .nonNullValues()
                        .toMap();
        return new BsAuctionKeywordBidFieldWriter(requiredFields, bsAuctionDataByBidId);
    }


    @Override
    public void write(KeywordBidGetItem bid, Long bidId) {
        writer.write(bid, bidId);
    }

    @Nullable
    private KeywordBidGetItemWriter getKeywordBidGetItemWriter(KeywordBidAnyFieldEnum keywordIdField,
                                                               KeywordBidGetItemWriter keywordIdWriter) {
        return !requiredFields.contains(keywordIdField) ? null : keywordIdWriter;
    }

    @Nullable
    private KeywordBidGetItemWriter getAuctionBidsWriter() {
        return getKeywordBidGetItemWriter(KeywordBidAnyFieldEnum.SEARCH_AUCTION_BIDS, (item, id) -> {
                    KeywordTrafaretData trafaretData = bsResultsById.get(id);

                    if (item.getSearch() == null) {
                        item.setSearch(new Search());
                    }

                    // заполняем AuctionBids = null, если параметр был запрошен, но нет данных
                    AuctionBids auctionBids = null;
                    if (trafaretData != null) {
                        auctionBids = new AuctionBids().withAuctionBidItems(StreamEx.of(trafaretData.getBidItems())
                                .map(t -> new AuctionBidItem()
                                        // из БК приходит 1_157_000, переводим в 116
                                        .withTrafficVolume(
                                                (int) Math.round(t.getPositionCtrCorrection() / TRAFFIC_VOLUME_MULTIPLIER))
                                        .withBid(getShowValue(t.getBid()).micros())
                                        .withPrice(getShowValue(t.getPrice()).micros()))
                                .filter(t -> t.getTrafficVolume() > 0)
                                // если после округления получилось два одинаковых TrafficVolume с разными ставками,
                                // берём тот, у которого меньше ставка
                                .sorted(comparing(AuctionBidItem::getTrafficVolume).reversed())
                                .collapse(
                                        (r1, r2) -> Objects.equals(r1.getTrafficVolume(), r2.getTrafficVolume()),
                                        (r1, r2) -> r1.getBid() < r2.getBid() ? r1 : r2)
                                .toList());
                    }
                    item.getSearch().setAuctionBids(FACTORY.createSearchAuctionBids(auctionBids));
                }
        );
    }
}
