package ru.yandex.direct.api.v5.entity.keywordbids.converter;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.ServingStatusEnum;
import com.yandex.direct.api.v5.keywordbids.KeywordBidActionResult;
import com.yandex.direct.api.v5.keywordbids.KeywordBidGetItem;
import com.yandex.direct.api.v5.keywordbids.Network;
import com.yandex.direct.api.v5.keywordbids.ObjectFactory;
import com.yandex.direct.api.v5.keywordbids.Search;
import one.util.streamex.StreamEx;

import ru.yandex.direct.api.v5.entity.keywordbids.KeywordBidAnyFieldEnum;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.entity.keywordbids.converter.KeywordBidGetItemWriterComposition.writerCompositionOf;

/**
 * Реализация {@link KeywordBidGetItemWriter}, заполняющая поля {@link KeywordBidGetItem} на основе данных, хранящихся в Директе
 */
@ParametersAreNonnullByDefault
public class DirectKeywordBidFieldWriter implements KeywordBidGetItemWriter {
    private static final ObjectFactory FACTORY = new ObjectFactory();

    private final Set<KeywordBidAnyFieldEnum> requiredFields;
    private final Map<Long, CompleteBidData<KeywordTrafaretData>> completeBidDataById;

    private final KeywordBidGetItemWriter writer;

    private DirectKeywordBidFieldWriter(Set<KeywordBidAnyFieldEnum> requiredFields,
                                        Map<Long, CompleteBidData<KeywordTrafaretData>> completeBidDataById) {
        this.requiredFields = requiredFields;
        this.completeBidDataById = completeBidDataById;
        writer = writerCompositionOf(
                getKeywordIdWriter(),
                getAdGroupIdWriter(),
                getCampaignIdWriter(),
                getSearchBidWriter(),
                getNetworkBidWriter(),
                getStrategyPriorityWriter(),
                getServingStatusWriter());
    }

    /**
     * @see DirectKeywordBidFieldWriter
     */
    static DirectKeywordBidFieldWriter createDirectKeywordBidFieldWriter(Set<KeywordBidAnyFieldEnum> requiredFields,
                                                                         Collection<CompleteBidData<KeywordTrafaretData>> completeBidData) {
        Map<Long, CompleteBidData<KeywordTrafaretData>> completeBidDataById = StreamEx.of(completeBidData)
                .mapToEntry(CompleteBidData::getBidId)
                .invert()
                .toMap();
        return new DirectKeywordBidFieldWriter(requiredFields, completeBidDataById);
    }


    @Override
    public void write(KeywordBidGetItem bid, Long bidId) {
        writer.write(bid, bidId);
    }

    private KeywordBidGetItemWriter getKeywordIdWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.KEYWORD_ID, KeywordBidActionResult::setKeywordId);
    }

    private KeywordBidGetItemWriter getAdGroupIdWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.AD_GROUP_ID,
                (item, id) -> item.setAdGroupId(getPhrase(id).getAdGroupId()));
    }

    private KeywordBidGetItemWriter getCampaignIdWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.CAMPAIGN_ID,
                (item, id) -> item.setCampaignId(getPhrase(id).getCampaignId()));
    }

    private KeywordBidGetItemWriter getSearchBidWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.SEARCH_BID, (item, id) -> {
            if (item.getSearch() == null) {
                item.setSearch(new Search());
            }
            item.getSearch().setBid(priceToExternalBid(id, getPhrase(id).getPrice()));
        });
    }

    private KeywordBidGetItemWriter getNetworkBidWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.NETWORK_BID, (item, id) -> {
            if (item.getNetwork() == null) {
                item.setNetwork(new Network());
            }
            item.getNetwork().setBid(priceToExternalBid(id, getPhrase(id).getPriceContext()));
        });
    }

    private KeywordBidGetItemWriter getStrategyPriorityWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.STRATEGY_PRIORITY,
                (item, id) -> item.setStrategyPriority(FACTORY.createKeywordBidGetItemStrategyPriority(
                        convertStrategyPriority(getPhrase(id).getAutobudgetPriority())))
        );
    }

    private KeywordBidGetItemWriter getServingStatusWriter() {
        return getItemWriter(KeywordBidAnyFieldEnum.SERVING_STATUS, (item, id) -> item
                .setServingStatus(getAdGroupByPhraseId(id).getBsRarelyLoaded()
                        ? ServingStatusEnum.RARELY_SERVED
                        : ServingStatusEnum.ELIGIBLE));
    }

    private AdGroup getAdGroupByPhraseId(Long id) {
        return completeBidDataById.get(id).getAdGroup();
    }

    private Bid getPhrase(Long id) {
        return completeBidDataById.get(id).getBid();
    }

    @Nullable
    private KeywordBidGetItemWriter getItemWriter(KeywordBidAnyFieldEnum keywordIdField,
                                                  KeywordBidGetItemWriter keywordIdWriter) {
        return !requiredFields.contains(keywordIdField) ? null : keywordIdWriter;
    }

    /**
     * Переводит ставку из внутреннего формата во внешний.
     * Нулевые ставки превращаются в минимальные для данной валюты
     *
     * @param id    id ключевого слова
     * @param price ставка
     * @return ставка во внешнем формате в микроденьгах
     */
    private Long priceToExternalBid(Long id, BigDecimal price) {
        return (price.compareTo(BigDecimal.ZERO) == 0) ?
                getKeywordBidCurrencyMinPriceMicro(id) :
                price.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    /**
     * Возвращает минимальную ставку в валюте кампании, которой принадлежит ключевое слово
     *
     * @param id id ключевого слова
     * @return минимальная ставка в микроденьгах
     */
    private Long getKeywordBidCurrencyMinPriceMicro(Long id) {
        Campaign campaign = completeBidDataById.get(id).getCampaign();
        Currency currency = campaign.getCurrency().getCurrency();
        AdGroup adGroup = completeBidDataById.get(id).getAdGroup();
        BigDecimal defaultPrice =
                (adGroup.getType() == AdGroupType.CPM_BANNER) ? currency.getMinCpmPrice() : currency.getMinPrice();
        return defaultPrice.multiply(Money.MICRO_MULTIPLIER).longValue();
    }
}
