package ru.yandex.direct.api.v5.entity.keywordbids.converter;

import java.util.EnumSet;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywordbids.BiddingRule;
import com.yandex.direct.api.v5.keywordbids.KeywordBidActionResult;
import com.yandex.direct.api.v5.keywordbids.KeywordBidSetAutoItem;
import com.yandex.direct.api.v5.keywordbids.NetworkByCoverage;
import com.yandex.direct.api.v5.keywordbids.SearchByTrafficVolume;
import com.yandex.direct.api.v5.keywordbids.SetAutoRequest;
import com.yandex.direct.api.v5.keywordbids.SetAutoResponse;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.core.entity.bids.container.BidTargetType;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetAutoNetworkByCoverage;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByPosition;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByTrafficVolume;
import ru.yandex.direct.validation.result.PathConverter;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;

@Component
@ParametersAreNonnullByDefault
public class SetAutoKeywordBidsConverter {

    private final ResultConverter resultConverter;

    public SetAutoKeywordBidsConverter(ResultConverter resultConverter) {
        this.resultConverter = resultConverter;
    }

    /**
     * @return преобразованный внешний запрос {@link SetAutoRequest}
     * в список {@link SetAutoBidItem} - core-представление запроса на изменение ставок
     * @see ru.yandex.direct.api.v5.entity.bids.converter.BidsHelperConverter#convertFromSetAutoRequest(com.yandex.direct.api.v5.bids.SetAutoRequest)
     */
    @Nonnull
    public List<SetAutoBidItem> convertFromSetAutoRequest(SetAutoRequest request) {
        if (request.getKeywordBids() == null) {
            return emptyList();
        }
        return StreamEx.of(request.getKeywordBids())
                .map(this::convertSetAutoItem)
                .toList();
    }

    private SetAutoBidItem convertSetAutoItem(KeywordBidSetAutoItem item) {
        BiddingRule biddingRule = item.getBiddingRule();
        EnumSet<BidTargetType> scopeList = EnumSet.noneOf(BidTargetType.class);

        SetAutoSearchByTrafficVolume searchByTrafficVolume = new SetAutoSearchByTrafficVolume();
        SearchByTrafficVolume externalSearch = biddingRule.getSearchByTrafficVolume();
        if (externalSearch != null) {
            scopeList.add(BidTargetType.SEARCH_BY_TRAFFIC_VOLUME);
            searchByTrafficVolume.withIncreasePercent(externalSearch.getIncreasePercent())
                    .withTargetTrafficVolume(externalSearch.getTargetTrafficVolume())
                    .withMaxBid(convertToDbPrice(externalSearch.getBidCeiling()));
        }

        SetAutoNetworkByCoverage networkByCoverage = new SetAutoNetworkByCoverage();
        NetworkByCoverage externalNetwork = biddingRule.getNetworkByCoverage();
        if (externalNetwork != null) {
            scopeList.add(BidTargetType.CONTEXT);
            networkByCoverage.withIncreasePercent(externalNetwork.getIncreasePercent())
                    .withContextCoverage(externalNetwork.getTargetCoverage())
                    .withMaxBid(convertToDbPrice(externalNetwork.getBidCeiling()));
        }

        return new SetAutoBidItem()
                .withCampaignId(item.getCampaignId())
                .withAdGroupId(item.getAdGroupId())
                .withId(item.getKeywordId())
                .withScope(scopeList)
                // инвариант: все сложные поля не null, поэтому заполняем пустой SetAutoSearchByPosition
                .withSearchByPosition(new SetAutoSearchByPosition())
                .withSearchByTrafficVolume(searchByTrafficVolume)
                .withNetworkByCoverage(networkByCoverage);
    }

    /**
     * @return внутренний {@link ApiResult}, преобразованный во внешнее представление {@link SetAutoResponse}
     * @see ru.yandex.direct.api.v5.entity.bids.converter.BidsHelperConverter#convertToSetAutoResponse(ApiResult, PathConverter)
     */
    @Nonnull
    public SetAutoResponse convertToSetAutoResponse(ApiResult<List<ApiResult<SetAutoBidItem>>> result,
                                                    PathConverter pathConverter) {
        List<KeywordBidActionResult> results =
                StreamEx.of(result.getResult())
                        .mapToEntry(r -> resultConverter.convertToGenericActionResult(r, pathConverter))
                        .mapKeyValue((r, actionResult) -> {
                            KeywordBidActionResult rr = new KeywordBidActionResult();
                            if (r.isSuccessful()) {
                                SetAutoBidItem bidItem = r.getResult();
                                rr.withKeywordId(bidItem.getId())
                                        .withAdGroupId(bidItem.getAdGroupId())
                                        .withCampaignId(bidItem.getCampaignId());
                            }
                            // getErrors() и getWarnings() под капотом лениво инициализируют списки ошибок и предупреждений
                            // проверяем списки на наличие элементов, чтобы не отдавать "Errors" и "Warnings", если ошибок не было
                            // See also ResultConverter.toKeywordBidActionResult()
                            if (!actionResult.getErrors().isEmpty()) {
                                rr.setErrors(actionResult.getErrors());
                            }
                            if (!actionResult.getWarnings().isEmpty()) {
                                rr.setWarnings(actionResult.getWarnings());
                            }
                            return rr;
                        })
                        .toList();
        return new SetAutoResponse()
                .withSetAutoResults(results);
    }
}
