package ru.yandex.direct.api.v5.entity.keywordbids.converter;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.SetBidsActionResult;
import com.yandex.direct.api.v5.keywordbids.KeywordBidActionResult;
import com.yandex.direct.api.v5.keywordbids.KeywordBidSetItem;
import com.yandex.direct.api.v5.keywordbids.SetRequest;
import com.yandex.direct.api.v5.keywordbids.SetResponse;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.bids.converter.BidsHelperConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.validation.result.PathConverter;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Входная точка для преобразования полученной информации о ставках в список {@link KeywordBidSetItem} для ответа
 * Работает аналогично {@link BidsHelperConverter}
 */

@Component
@ParametersAreNonnullByDefault
public class SetKeywordBidsConverter {

    private final ResultConverter resultConverter;

    public SetKeywordBidsConverter(ResultConverter resultConverter) {
        this.resultConverter = resultConverter;
    }

    @Nonnull
    public SetResponse convertSetBidsResponse(ApiResult<List<ApiResult<SetBidItem>>> result,
                                              PathConverter pathConverter) {
        List<KeywordBidActionResult> actionResults = StreamEx.of(result)
                .map(rlist -> StreamEx.of(rlist.getResult())
                        .map(r -> convertToBidActionResult(r, pathConverter))
                        .toList())
                .toFlatList(Function.identity());

        return new SetResponse().withSetResults(actionResults);
    }

    private KeywordBidActionResult convertToBidActionResult(ApiResult<SetBidItem> result,
                                                            PathConverter apiPathConverter) {
        return toBidActionResult(convertToSetBidsActionResult(result, apiPathConverter));
    }

    private SetBidsActionResult convertToSetBidsActionResult(ApiResult<SetBidItem> result,
                                                             PathConverter apiPathConverter) {
        return resultConverter.toActionResult(result, apiPathConverter, SetBidsActionResult::new,
                (actionResult, bid) -> actionResult.withId(bid.getId())
                        .withAdGroupId(bid.getAdGroupId())
                        .withCampaignId(bid.getCampaignId()));
    }

    private KeywordBidActionResult toBidActionResult(SetBidsActionResult setBidsActionResult) {
        KeywordBidActionResult bidActionResult = new KeywordBidActionResult()
                .withKeywordId(setBidsActionResult.getId())
                .withCampaignId(setBidsActionResult.getCampaignId())
                .withAdGroupId(setBidsActionResult.getAdGroupId());
        // getErrors() и getWarnings() под капотом лениво инициализируют списки ошибок и предупреждений
        // проверяем списки на наличие элементов, чтобы не отдавать "Errors" и "Warnings", если ошибок не было
        if (!setBidsActionResult.getErrors().isEmpty()) {
            bidActionResult.setErrors(setBidsActionResult.getErrors());
        }
        if (!setBidsActionResult.getWarnings().isEmpty()) {
            bidActionResult.setWarnings(setBidsActionResult.getWarnings());
        }
        return bidActionResult;
    }

    /**
     * Конвертирует запрос из объектного представления JSON'а во внутренние сущности-контейнеры
     *
     * @param setBidsRequest Запрос, пришедший от пользователя
     * @return Набор бидов, содержащих данные запроса пользователя
     */
    @Nonnull
    public List<SetBidItem> convertSetBidsRequest(SetRequest setBidsRequest) {
        if (setBidsRequest.getKeywordBids() == null) {
            return new ArrayList<>();
        }
        return mapList(setBidsRequest.getKeywordBids(), this::convertSetBidsItem);
    }

    @Nullable
    private SetBidItem convertSetBidsItem(@Nullable KeywordBidSetItem item) {
        if (item == null) {
            return null;
        }

        return new SetBidItem()
                .withShowConditionType(ShowConditionType.KEYWORD)
                .withId(item.getKeywordId())
                .withAdGroupId(item.getAdGroupId())
                .withCampaignId(item.getCampaignId())
                .withPriceSearch(convertToDbPrice(item.getSearchBid()))
                .withPriceContext(convertToDbPrice(item.getNetworkBid()))
                .withAutobudgetPriority(convertStrategyPriority(item.getStrategyPriority()));
    }
}
