package ru.yandex.direct.api.v5.entity.keywordbids.delegate;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Ordering;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import com.yandex.direct.api.v5.keywordbids.GetRequest;
import com.yandex.direct.api.v5.keywordbids.GetResponse;
import com.yandex.direct.api.v5.keywordbids.KeywordBidGetItem;
import com.yandex.direct.api.v5.keywordbids.KeywordBidsSelectionCriteria;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.keywordbids.KeywordBidAnyFieldEnum;
import ru.yandex.direct.api.v5.entity.keywordbids.converter.GetKeywordBidsConverter;
import ru.yandex.direct.api.v5.entity.keywordbids.service.GetKeywordBidsValidationService;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.bids.container.CompleteBidData;
import ru.yandex.direct.core.entity.bids.container.KeywordBidDynamicData;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.bids.service.KeywordBidDynamicDataService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.keyword.model.ServingStatus;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.primitives.Longs.asList;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class GetKeywordBidsDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, KeywordBidAnyFieldEnum, ShowConditionSelectionCriteria, CompleteBidData<KeywordTrafaretData>> {
    private static final Logger logger = LoggerFactory.getLogger(GetKeywordBidsDelegate.class);
    private static final Ordering<TrafaretBidItem> TRAFARET_BID_ITEM_REVERSED_ORDERING =
            Ordering.from(Comparator.comparing(TrafaretBidItem::getPositionCtrCorrection).reversed());

    private final GetKeywordBidsValidationService getKeywordBidsValidationService;
    // todo maxlog (DIRECT-78875): то, что это поле не используется, выглядит крайне подозрительно
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final BidService bidService;
    private final KeywordBidDynamicDataService keywordBidDynamicDataService;
    private final ResultConverter resultConverter;

    /**
     * Значения trafficVolume, которые отдаём пользователям
     */
    private static final Set<Long> TRAFFIC_VOLUMES_FOR_SHOW =
            new HashSet<>(asList(5_0000,
                    10_0000,
                    15_0000,
                    65_0000,
                    70_0000,
                    75_0000,
                    80_0000,
                    85_0000,
                    86_0000,
                    87_0000,
                    88_0000,
                    89_0000,
                    90_0000,
                    91_0000,
                    92_0000,
                    93_0000,
                    94_0000,
                    95_0000,
                    96_0000,
                    97_0000,
                    98_0000,
                    99_0000,
                    100_0000,
                    101_0000,
                    102_0000,
                    103_0000,
                    104_0000,
                    105_0000,
                    106_0000,
                    107_0000,
                    108_0000,
                    109_0000,
                    110_0000,
                    111_0000,
                    112_0000,
                    113_0000,
                    114_0000,
                    115_0000,
                    116_0000,
                    117_0000,
                    118_0000,
                    119_0000,
                    120_0000));

    @Autowired
    public GetKeywordBidsDelegate(ApiAuthenticationSource auth,
                                  GetKeywordBidsValidationService getKeywordBidsValidationService,
                                  CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                  BidService bidService,
                                  KeywordBidDynamicDataService keywordBidDynamicDataService,
                                  ResultConverter resultConverter) {
        // так как все валидации делаются на уровне API, можем PathConverter брать прямо из ValidationService
        super(getKeywordBidsValidationService.pathConverter(), auth);
        this.getKeywordBidsValidationService = getKeywordBidsValidationService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.bidService = bidService;
        this.keywordBidDynamicDataService = keywordBidDynamicDataService;
        this.resultConverter = resultConverter;
    }


    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        return getKeywordBidsValidationService.validate(externalRequest);
    }

    @Override
    public List<Long> returnedCampaignIds(ApiResult<List<CompleteBidData<KeywordTrafaretData>>> result) {
        return mapList(result.getResult(), data -> data.getCampaign().getId());
    }

    @Override
    public ShowConditionSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        KeywordBidsSelectionCriteria bidsSelectionCriteria = externalRequest.getSelectionCriteria();
        return new ShowConditionSelectionCriteria()
                .withShowConditionIds(bidsSelectionCriteria.getKeywordIds())
                .withAdGroupIds(bidsSelectionCriteria.getAdGroupIds())
                .withCampaignIds(bidsSelectionCriteria.getCampaignIds())
                .withServingStatuses(Collections.singleton(toServingStatus(bidsSelectionCriteria.getServingStatuses())));
    }

    @Override
    public Set<KeywordBidAnyFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return Stream.of(
                externalRequest.getFieldNames().stream().map(KeywordBidAnyFieldEnum::fromKeywordBidFieldEnum),
                externalRequest.getSearchFieldNames().stream()
                        .map(KeywordBidAnyFieldEnum::fromKeywordBidSearchFieldEnum),
                externalRequest.getNetworkFieldNames().stream()
                        .map(KeywordBidAnyFieldEnum::fromKeywordBidNetworkFieldEnum)
        ).flatMap(Function.identity()).collect(toSet());
    }

    @Override
    public List<CompleteBidData<KeywordTrafaretData>> get(
            GenericGetRequest<KeywordBidAnyFieldEnum, ShowConditionSelectionCriteria> getRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        Long operatorUid = auth.getOperator().getUid();

        ShowConditionSelectionCriteria selection = getRequest.getSelectionCriteria();
        List<Bid> bids = bidService.getBids(clientId, operatorUid, selection, getRequest.getLimitOffset());

        Set<KeywordBidAnyFieldEnum> requestedFields = getRequest.getRequestedFields();
        boolean hasPokazometerFields = hasPokazometerFields(requestedFields);
        boolean hasBsAuctionFields = hasBsAuctionFields(requestedFields);

        boolean safePokazometer = true;
        //noinspection ConstantConditions
        Collection<CompleteBidData<KeywordTrafaretData>> dynamicData = keywordBidDynamicDataService
                .getCompleteBidDataTrafaretFormat(clientId, bids, hasPokazometerFields, hasBsAuctionFields,
                        safePokazometer);

        checkState(dynamicData.size() == bids.size(),
                "Mismatch collections of bids (size: %d) and dynamicBidData (size: %d)",
                bids.size(), dynamicData.size());

        for (CompleteBidData<KeywordTrafaretData> bidData : dynamicData) {
            KeywordBidDynamicData<KeywordTrafaretData> singleDynData = bidData.getDynamicData();
            if (singleDynData == null) {
                // Не получали данные по фразе, например, потому что она в группе со статусом "Мало показов"
                // Или не запрашивались данные ни Показометра, ни Торгов, потому что пользователь не просил
                continue;
            }
            KeywordTrafaretData bsAuctionData = singleDynData.getBsAuctionData();
            if (bsAuctionData == null) {
                // В запросе не было полей из Торгов
                continue;
            }
            // Оставляем только элементы с правильными trafficVolume
            List<TrafaretBidItem> newTrafaretItems = sortAndFilterTrafaretItems(bsAuctionData.getBidItems());
            if (newTrafaretItems.isEmpty() && !bsAuctionData.getBidItems().isEmpty()) {
                logger.warn("No trafaret items suitable for API. KeywordId {}", bsAuctionData.getKeyword().getId());
            }
            bsAuctionData.withBidItems(newTrafaretItems);
        }
        return StreamEx.of(dynamicData)
                .sortedBy(dynData -> dynData.getBid().getId())
                .toList();
    }

    /**
     * Возвращает список, содержащий только элементы с объёмом трафика, указанным в {@link #TRAFFIC_VOLUMES_FOR_SHOW},
     * и максимальным значением trafficVolume.
     * Элементы в результирующем списке упорядоченны по убыванию {@code positionCtrCorrection}
     */
    private List<TrafaretBidItem> sortAndFilterTrafaretItems(List<TrafaretBidItem> bidItems) {
        if (bidItems.isEmpty()) {
            return emptyList();
        }
        // Из ядра нам должны приходить списки, упорядоченные по убыванию positionCtrCorrection
        boolean isOrdered = TRAFARET_BID_ITEM_REVERSED_ORDERING.isOrdered(bidItems);
        List<TrafaretBidItem> orderedBidItems;
        if (isOrdered) {
            orderedBidItems = bidItems;
        } else {
            orderedBidItems = TRAFARET_BID_ITEM_REVERSED_ORDERING.sortedCopy(bidItems);
        }

        List<TrafaretBidItem> result = new ArrayList<>();
        // безусловно добавляем максимальный элемент
        result.add(orderedBidItems.get(0));
        // прочие элементы просеиваем через TRAFFIC_VOLUMES_FOR_SHOW
        for (int i = 1; i < orderedBidItems.size(); i++) {
            TrafaretBidItem t = orderedBidItems.get(i);
            if (TRAFFIC_VOLUMES_FOR_SHOW.contains(t.getPositionCtrCorrection())) {
                result.add(t);
            }
        }
        return result;
    }

    @Override
    public GetResponse convertGetResponse(List<CompleteBidData<KeywordTrafaretData>> dynamicBidData,
                                          Set<KeywordBidAnyFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        List<KeywordBidGetItem> items = GetKeywordBidsConverter
                .convertCompleteBidDataToKeywordBidGetItems(requestedFields, dynamicBidData);

        GetResponse getResponse = new GetResponse();
        if (!items.isEmpty()) {
            getResponse.withKeywordBids(items);
        }
        return getResponse.withLimitedBy(limitedBy);
    }

    @Nullable
    private static ServingStatus toServingStatus(List<ServingStatusEnum> servingStatusEnums) {
        List<ServingStatus> servingStatuses = StreamEx.of(servingStatusEnums)
                .map(ServingStatusEnum::value)
                .map(ServingStatus::valueOf)
                .toList();

        // Сейчас возможны два взаимоисключающих статуса: ELIGIBLE и RARELY_SERVED.
        // Если заданы оба или не задан ни один, то возвращаем все ставки
        if (servingStatuses.size() == 1) {
            return servingStatuses.get(0);
        } else {
            return null;
        }
    }

    private boolean hasPokazometerFields(Set<KeywordBidAnyFieldEnum> requestedFields) {
        return requestedFields.contains(KeywordBidAnyFieldEnum.NETWORK_COVERAGE);
    }

    private boolean hasBsAuctionFields(Set<KeywordBidAnyFieldEnum> requestedFields) {
        return requestedFields.contains(KeywordBidAnyFieldEnum.SEARCH_AUCTION_BIDS);
    }
}
