package ru.yandex.direct.api.v5.entity.keywords;

import java.util.Map;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.keywords.AddRequest;
import com.yandex.direct.api.v5.keywords.AddResponse;
import com.yandex.direct.api.v5.keywords.DeleteRequest;
import com.yandex.direct.api.v5.keywords.DeleteResponse;
import com.yandex.direct.api.v5.keywords.GetRequest;
import com.yandex.direct.api.v5.keywords.GetResponse;
import com.yandex.direct.api.v5.keywords.KeywordsPort;
import com.yandex.direct.api.v5.keywords.ResumeRequest;
import com.yandex.direct.api.v5.keywords.ResumeResponse;
import com.yandex.direct.api.v5.keywords.SuspendRequest;
import com.yandex.direct.api.v5.keywords.SuspendResponse;
import com.yandex.direct.api.v5.keywords.UpdateRequest;
import com.yandex.direct.api.v5.keywords.UpdateResponse;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.entity.GenericApiService;
import ru.yandex.direct.api.v5.entity.keywords.delegate.AddKeywordsDelegate;
import ru.yandex.direct.api.v5.entity.keywords.delegate.DeleteKeywordsDelegate;
import ru.yandex.direct.api.v5.entity.keywords.delegate.GetKeywordsDelegate;
import ru.yandex.direct.api.v5.entity.keywords.delegate.ResumeKeywordsDelegate;
import ru.yandex.direct.api.v5.entity.keywords.delegate.SuspendKeywordsDelegate;
import ru.yandex.direct.api.v5.entity.keywords.delegate.UpdateKeywordsDelegate;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.api.v5.ws.annotation.ApiRequest;
import ru.yandex.direct.api.v5.ws.annotation.ApiResponse;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceEndpoint;
import ru.yandex.direct.api.v5.ws.annotation.ApiServiceType;
import ru.yandex.direct.api.v5.ws.annotation.ServiceType;
import ru.yandex.direct.core.entity.bids.validation.BidsDefects;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefectIds;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseDefectIds;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.i18n.types.Identity;
import ru.yandex.direct.validation.defect.ids.CollectionDefectIds;
import ru.yandex.direct.validation.defect.ids.StringDefectIds;
import ru.yandex.direct.validation.result.DefectIds;

import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidMustBeNotGreaterThan;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidMustBeNotLessThan;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidWontBeAcceptedInCaseOfAutoBudgetStrategy;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidWontBeAcceptedInCaseOfContextDependsSearch;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.bidWontBeAcceptedInCaseOfContextIsServingOff;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.priorityWontBeAcceptedNotAutoBudgetStrategy;

/**
 * Сервис по работе с ключевыми фразами и автотаргетингами.
 *
 * @see <a href=https://tech.yandex.ru/direct/doc/ref-v5/keywords/keywords-docpage/>Keywords API tech docs</a>
 */
@ApiServiceEndpoint
@ApiServiceType(type = ServiceType.CLIENT)
public class KeywordsEndpoint implements KeywordsPort {
    private static final String SERVICE_NAME = "keywords";
    public static final DefectPresentationsHolder KEYWORDS_CUSTOM_DEFECT_PRESENTATIONS =
            DefectPresentationsHolder.builder()
                    .register(DefectIds.OBJECT_NOT_FOUND,
                            KeywordsDefectTypes.keywordNotFound())
                    .register(KeywordDefectIds.Gen.AD_GROUP_NOT_FOUND,
                            KeywordsDefectTypes.adGroupNotFound())
                    .register(StringDefectIds.LENGTH_CANNOT_BE_MORE_THAN_MAX,
                            t -> DefectTypes.lengthOfFieldValueMustNotExceed(t.getMaxLength()))
                    .register(CollectionDefectIds.Gen.MUST_NOT_CONTAIN_DUPLICATED_ELEMENTS,
                            KeywordsDefectTypes.keywordIsDuplicated())
                    .register(KeywordDefectIds.Keyword.MAX_KEYWORDS_PER_AD_GROUP_EXCEEDED,
                            t -> KeywordsDefectTypes.maxKeywordsInAdGroup(t.getMaxKeywords()))
                    .register(BidsDefects.CurrencyAmountDefects.SEARCH_PRICE_IS_NOT_GREATER_THAN_MIN,
                            t -> bidMustBeNotLessThan(t.getMoneyValue().getCurrencyCode().getTranslation().longForm(),
                                    Identity.of(t.getMoneyValue().micros())))
                    .register(BidsDefects.CurrencyAmountDefects.SEARCH_PRICE_IS_NOT_SMALLER_THAN_MAX,
                            t -> bidMustBeNotGreaterThan(
                                    t.getMoneyValue().getCurrencyCode().getTranslation().longForm(),
                                    Identity.of(t.getMoneyValue().micros())))
                    .register(BidsDefects.CurrencyAmountDefects.CPM_PRICE_IS_NOT_GREATER_THAN_MIN,
                            t -> bidMustBeNotLessThan(t.getMoneyValue().getCurrencyCode().getTranslation().longForm(),
                                    Identity.of(t.getMoneyValue().micros())))
                    .register(BidsDefects.CurrencyAmountDefects.CPM_PRICE_IS_NOT_SMALLER_THAN_MAX,
                            t -> bidMustBeNotGreaterThan(
                                    t.getMoneyValue().getCurrencyCode().getTranslation().longForm(),
                                    Identity.of(t.getMoneyValue().micros())))
                    .register(PhraseDefectIds.String.MINUS_WORD_DELETE_PLUS_WORD,
                            t -> KeywordsDefectTypes.keywordAsMinusWord(t.getAllInvalidSubstrings()))
                    .register(PhraseDefectIds.String.NOT_SINGLE_MINUS_WORD,
                            t -> KeywordsDefectTypes.minusWordsNoMinusPhrases())
                    .register(PhraseDefectIds.String.ILLEGAL_CHARACTERS,
                            KeywordsDefectTypes.keywordWithInvalidSymbols())
                    .register(PhraseDefectIds.Gen.NO_PLUS_WORDS, KeywordsDefectTypes.keywordContainsOnlyMinusWords())
                    .register(PhraseDefectIds.Gen.INVALID_BRACKETS, KeywordsDefectTypes.invalidBracketsInKeyword())
                    .register(PhraseDefectIds.Gen.PLUS_MARK_IN_BRACKETS, KeywordsDefectTypes.plusMarkInBrackets())
                    .register(PhraseDefectIds.String.TOO_LONG_WORD,
                            t -> KeywordsDefectTypes.keywordMaxLength(t.getMaxLength(), t.getAllInvalidSubstrings()))
                    .register(PhraseDefectIds.String.TOO_LONG_MINUS_WORD,
                            t -> KeywordsDefectTypes.minusWordMaxLength(t.getMaxLength(), t.getAllInvalidSubstrings()))
                    .register(PhraseDefectIds.String.TOO_LONG_KEYWORD,
                            t -> KeywordsDefectTypes.keywordsTooLong(t.getMaxLength()))
                    .register(PhraseDefectIds.String.TOO_MANY_WORDS,
                            t -> KeywordsDefectTypes.keywordMaxWords(t.getMaxWords()))
                    .register(PhraseDefectIds.Gen.ONLY_STOP_WORDS, KeywordsDefectTypes.containsOnlyStopWords())
                    .register(PhraseDefectIds.Gen.INVALID_QUOTES, KeywordsDefectTypes.invalidQuotes())
                    .register(StringDefectIds.CANNOT_BE_EMPTY, DefectTypes.absentValueInField())
                    .register(BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF,
                            t -> bidWontBeAcceptedInCaseOfContextIsServingOff())
                    .register(BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_SEARCH_IS_SWITCHED_OFF,
                            t -> KeywordsDefectTypes.bidWontBeAcceptedSearchIsSwitchedOff())
                    .register(BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY,
                            t -> bidWontBeAcceptedInCaseOfAutoBudgetStrategy())
                    .register(BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY,
                            t -> bidWontBeAcceptedInCaseOfAutoBudgetStrategy())
                    .register(BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NOT_DIFFERENT_PLACES,
                            t -> bidWontBeAcceptedInCaseOfContextDependsSearch())
                    .register(BidsDefects.Ids.PRIORITY_WONT_BE_ACCEPTED_IN_CASE_OF_NOT_AUTO_BUDGET_STRATEGY,
                            t -> priorityWontBeAcceptedNotAutoBudgetStrategy())
                    .register(CampaignDefectIds.Gen.INCONSISTENT_CAMPAIGN_TYPE,
                            KeywordsDefectTypes.autotargetingCategoriesForbiddenInThisAdGroup())
                    .build();

    private final GenericApiService genericApiService;
    private final AddKeywordsDelegate addKeywordsDelegate;
    private final UpdateKeywordsDelegate updateKeywordsDelegate;
    private final SuspendKeywordsDelegate suspendKeywordsDelegate;
    private final ResumeKeywordsDelegate resumeKeywordsDelegate;
    private final DeleteKeywordsDelegate deleteKeywordsDelegate;
    private final GetKeywordsDelegate getKeywordsDelegate;

    @Autowired
    public KeywordsEndpoint(
            GenericApiService genericApiService,
            AddKeywordsDelegate addKeywordsDelegate,
            UpdateKeywordsDelegate updateKeywordsDelegate,
            SuspendKeywordsDelegate suspendKeywordsDelegate,
            ResumeKeywordsDelegate resumeKeywordsDelegate,
            DeleteKeywordsDelegate deleteKeywordsDelegate,
            GetKeywordsDelegate getKeywordsDelegate) {
        this.genericApiService = genericApiService;
        this.addKeywordsDelegate = addKeywordsDelegate;
        this.updateKeywordsDelegate = updateKeywordsDelegate;
        this.suspendKeywordsDelegate = suspendKeywordsDelegate;
        this.resumeKeywordsDelegate = resumeKeywordsDelegate;
        this.deleteKeywordsDelegate = deleteKeywordsDelegate;
        this.getKeywordsDelegate = getKeywordsDelegate;
    }

    public static Map<String, String> getKeywordsCommonDict() {
        return ImmutableMap.<String, String>builder()
                .putAll(ApiPathConverter.getCommonDict())
                .put(Keyword.PHRASE.name(), "Keyword")
                // Маппинг автотаргетинга технический, наружу отдаваться не должен
                .put(Keyword.IS_AUTOTARGETING.name(), "Autotargeting")
                .put(Keyword.HREF_PARAM1.name(), "UserParam1")
                .put(Keyword.HREF_PARAM2.name(), "UserParam2")
                .build();
    }

    @PreAuthorizeRead
    @ApiMethod(service = SERVICE_NAME, operation = "get")
    @ApiResponse
    @Override
    public GetResponse get(@ApiRequest GetRequest parameters) {
        return genericApiService.doAction(getKeywordsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "add")
    @ApiResponse
    @Override
    public AddResponse add(@ApiRequest AddRequest parameters) {
        return genericApiService.doAction(addKeywordsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "update")
    @ApiResponse
    @Override
    public UpdateResponse update(@ApiRequest UpdateRequest parameters) {
        return genericApiService.doAction(updateKeywordsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "delete")
    @ApiResponse
    @Override
    public DeleteResponse delete(@ApiRequest DeleteRequest deleteRequest) {
        return genericApiService.doAction(deleteKeywordsDelegate, deleteRequest);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "suspend")
    @ApiResponse
    @Override
    public SuspendResponse suspend(@ApiRequest SuspendRequest parameters) {
        return genericApiService.doAction(suspendKeywordsDelegate, parameters);
    }

    @PreAuthorizeWrite
    @ApiMethod(service = SERVICE_NAME, operation = "resume")
    @ApiResponse
    @Override
    public ResumeResponse resume(@ApiRequest ResumeRequest parameters) {
        return genericApiService.doAction(resumeKeywordsDelegate, parameters);
    }

    private static RuntimeException notImplementedYet() {
        return new UnsupportedOperationException("Not implemented yet.");
    }
}
