package ru.yandex.direct.api.v5.entity.keywords.converter;


import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.yandex.direct.api.v5.general.ServingStatusEnum;
import com.yandex.direct.api.v5.keywords.KeywordStateSelectionEnum;
import com.yandex.direct.api.v5.keywords.KeywordStatusSelectionEnum;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bids.container.ShowConditionStateSelection;
import ru.yandex.direct.core.entity.bids.container.ShowConditionStatusSelection;
import ru.yandex.direct.core.entity.keyword.model.ServingStatus;
import ru.yandex.direct.utils.converter.Converters;

@ParametersAreNonnullByDefault
@Component
public class KeywordsGetRequestConverter {
    private static ZoneId localTimezone;

    public KeywordsGetRequestConverter(@Value("${timezone}") String timezone) {
        localTimezone = ZoneId.of(timezone);
    }

    private static final Map<KeywordStateSelectionEnum, ShowConditionStateSelection> EXTERNAL_TO_INTERNAL_STATES =
            ImmutableMap.<KeywordStateSelectionEnum, ShowConditionStateSelection>builder()
                    .put(KeywordStateSelectionEnum.OFF, ShowConditionStateSelection.OFF)
                    .put(KeywordStateSelectionEnum.ON, ShowConditionStateSelection.ON)
                    .put(KeywordStateSelectionEnum.SUSPENDED, ShowConditionStateSelection.SUSPENDED)
                    .build();

    private static final Map<KeywordStatusSelectionEnum, ShowConditionStatusSelection> EXTERNAL_TO_INTERNAL_STATUSES =
            ImmutableMap.<KeywordStatusSelectionEnum, ShowConditionStatusSelection>builder()
                    .put(KeywordStatusSelectionEnum.ACCEPTED, ShowConditionStatusSelection.ACCEPTED)
                    .put(KeywordStatusSelectionEnum.DRAFT, ShowConditionStatusSelection.DRAFT)
                    .put(KeywordStatusSelectionEnum.REJECTED, ShowConditionStatusSelection.REJECTED)
                    .build();

    private static final Map<ServingStatusEnum, ServingStatus> EXTERNAL_TO_INTERNAL_SERVING_STATUSES =
            ImmutableMap.<ServingStatusEnum, ServingStatus>builder()
                    .put(ServingStatusEnum.ELIGIBLE, ServingStatus.ELIGIBLE)
                    .put(ServingStatusEnum.RARELY_SERVED, ServingStatus.RARELY_SERVED)
                    .build();

    public Set<ShowConditionStateSelection> convertStates(List<KeywordStateSelectionEnum> externalStates) {
        return Converters.immutableSetConverter(EXTERNAL_TO_INTERNAL_STATES).convert(externalStates);
    }

    public Set<ShowConditionStatusSelection> convertStatuses(List<KeywordStatusSelectionEnum> externalStatuses) {
        return Converters.immutableSetConverter(EXTERNAL_TO_INTERNAL_STATUSES).convert(externalStatuses);
    }

    public Set<ServingStatus> convertServingStatuses(List<ServingStatusEnum> externalExtStatuses) {
        return Converters.immutableSetConverter(EXTERNAL_TO_INTERNAL_SERVING_STATUSES).convert(externalExtStatuses);
    }

    @Nullable
    public LocalDateTime convertModifiedSince(@Nullable String modifiedSince) {
        if (modifiedSince == null) {
            return null;
        }
        ZonedDateTime zdt = ZonedDateTime.parse(modifiedSince, DateTimeFormatter.ISO_OFFSET_DATE_TIME);
        return zdt.withZoneSameInstant(localTimezone).toLocalDateTime();
    }
}

