package ru.yandex.direct.api.v5.entity.keywords.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.ServingStatusEnum;
import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.Statistics;
import com.yandex.direct.api.v5.general.StatusEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import com.yandex.direct.api.v5.keywords.ObjectFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.keywords.container.KeywordsGetContainer;
import ru.yandex.direct.api.v5.entity.keywords.container.StatValueAggregatorItem;
import ru.yandex.direct.api.v5.entity.keywords.service.StatisticService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.model.StatusModerate;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.common.RelevanceMatchCategoriesConverter.autotargetingCategoriesFromCore;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class KeywordsGetResponseConverter {
    private static final ObjectFactory FACTORY = new ObjectFactory();
    private final AdGroupService adGroupService;
    private final ClientService clientService;
    private final StatisticService statisticService;

    @Autowired
    KeywordsGetResponseConverter(AdGroupService adGroupService, ClientService clientService,
                                 StatisticService statisticService) {
        this.adGroupService = adGroupService;
        this.clientService = clientService;
        this.statisticService = statisticService;
    }

    public List<KeywordGetItem> convert(List<KeywordsGetContainer> keywordsGetContainers, ClientId clientId,
                                        Boolean requestedStat) {
        Client client = checkNotNull(clientService.getClient(clientId));
        Currency currency = client.getWorkCurrency().getCurrency();
        List<Long> adGroupIds = mapList(keywordsGetContainers, KeywordsGetContainer::getAdGroupId);
        List<AdGroup> adGroups = adGroupService.getAdGroups(clientId, adGroupIds);
        Map<Long, AdGroup> adGroupsMap = adGroups.stream().collect(toMap(AdGroup::getId, adGroup -> adGroup));
        Map<Long, StatValueAggregatorItem> showConditionStatistics =
                requestedStat ? statisticService.getPhraseStatistics(keywordsGetContainers) : null;
        return mapList(
                keywordsGetContainers,
                keywordsGetContainer -> convertItem(keywordsGetContainer, adGroupsMap, currency,
                        showConditionStatistics));
    }

    private KeywordGetItem convertItem(KeywordsGetContainer keywordsGetContainer, Map<Long, AdGroup> adGroupsMap,
                                       Currency currency,
                                       @Nullable Map<Long, StatValueAggregatorItem> showConditionStatistics) {
        if (keywordsGetContainer.hasRelevanceMatch()) {
            return convertRelevanceMatch(keywordsGetContainer, adGroupsMap, currency, showConditionStatistics);
        } else {
            return convertKeyword(keywordsGetContainer, adGroupsMap, currency, showConditionStatistics);
        }
    }

    private KeywordGetItem convertRelevanceMatch(KeywordsGetContainer keywordsGetContainer,
                                                 Map<Long, AdGroup> adGroupsMap, Currency currency,
                                                 @Nullable Map<Long, StatValueAggregatorItem> showConditionStat) {
        RelevanceMatch relevanceMatch = keywordsGetContainer.getRelevanceMatch();
        AdGroup adGroup = checkNotNull(adGroupsMap.get(relevanceMatch.getAdGroupId()));

        KeywordGetItem keywordGetItem = FACTORY.createKeywordGetItem().withId(relevanceMatch.getId());
        if (!adGroup.getBsRarelyLoaded()) {
            itemWithExternalStatistics(keywordGetItem, showConditionStat);
        }
        return keywordGetItem
                .withKeyword("---autotargeting")
                .withAdGroupId(relevanceMatch.getAdGroupId())
                .withCampaignId(relevanceMatch.getCampaignId())
                .withUserParam1(FACTORY.createKeywordGetItemUserParam1(relevanceMatch.getHrefParam1()))
                .withUserParam2(FACTORY.createKeywordGetItemUserParam2(relevanceMatch.getHrefParam2()))
                .withBid(priceToExternalBid(relevanceMatch.getPrice(), currency, adGroup.getType()))
                .withContextBid(priceToExternalBid(relevanceMatch.getPriceContext(), currency, adGroup.getType()))
                .withStrategyPriority(FACTORY.createKeywordGetItemStrategyPriority(
                        convertStrategyPriority(relevanceMatch.getAutobudgetPriority())))
                .withState(calcRelevanceMatchState(relevanceMatch.getIsSuspended()))
                .withStatus(StatusEnum.ACCEPTED)
                .withServingStatus(externalServingStatus(adGroup.getBsRarelyLoaded()))
                .withProductivity(FACTORY.createKeywordGetItemProductivity(null))
                .withAutotargetingCategories(FACTORY.createKeywordGetItemAutotargetingCategories(
                        autotargetingCategoriesFromCore(relevanceMatch.getRelevanceMatchCategories(),
                                adGroup.getType())));
    }

    private KeywordGetItem convertKeyword(KeywordsGetContainer keywordsGetContainer, Map<Long, AdGroup> adGroupsMap,
                                          Currency currency,
                                          @Nullable Map<Long, StatValueAggregatorItem> showConditionStat) {
        Keyword keyword = keywordsGetContainer.getKeyword();
        AdGroup adGroup = checkNotNull(adGroupsMap.get(keyword.getAdGroupId()));
        KeywordGetItem keywordGetItem = FACTORY.createKeywordGetItem().withId(keyword.getId());
        if (!adGroup.getBsRarelyLoaded()) {
            itemWithExternalStatistics(keywordGetItem, showConditionStat);
        }
        return keywordGetItem
                .withKeyword(keyword.getPhrase())
                .withAdGroupId(keyword.getAdGroupId())
                .withCampaignId(keyword.getCampaignId())
                .withUserParam1(FACTORY.createKeywordGetItemUserParam1(keyword.getHrefParam1()))
                .withUserParam2(FACTORY.createKeywordGetItemUserParam2(keyword.getHrefParam2()))
                .withBid(priceToExternalBid(keyword.getPrice(), currency, adGroup.getType()))
                .withContextBid(priceToExternalBid(keyword.getPriceContext(), currency, adGroup.getType()))
                .withStrategyPriority(FACTORY.createKeywordGetItemStrategyPriority(
                        convertStrategyPriority(keyword.getAutobudgetPriority())))
                .withState(calcKeywordState(keyword.getIsSuspended(), keyword.getStatusModerate()))
                .withStatus(calcKeywordStatus(keyword.getStatusModerate()))
                .withServingStatus(externalServingStatus(adGroup.getBsRarelyLoaded()))
                .withProductivity(FACTORY.createKeywordGetItemProductivity(null));
    }

    private static StateEnum calcKeywordState(Boolean isSuspended, StatusModerate statusModerate) {
        StateEnum externalState = StateEnum.UNKNOWN;

        if (isSuspended) {
            externalState = StateEnum.SUSPENDED;
        } else if (statusModerate == StatusModerate.YES) {
            externalState = StateEnum.ON;
        } else if (statusModerate == StatusModerate.NEW || statusModerate == StatusModerate.NO) {
            externalState = StateEnum.OFF;
        }
        return externalState;
    }

    private static StatusEnum calcKeywordStatus(StatusModerate statusModerate) {
        StatusEnum externalStatus = StatusEnum.UNKNOWN;
        if (statusModerate == StatusModerate.NEW) {
            externalStatus = StatusEnum.DRAFT;
        } else if (statusModerate == StatusModerate.YES) {
            externalStatus = StatusEnum.ACCEPTED;
        } else if (statusModerate == StatusModerate.NO) {
            externalStatus = StatusEnum.REJECTED;
        }
        return externalStatus;
    }

    private static StateEnum calcRelevanceMatchState(Boolean isSuspended) {
        if (isSuspended) {
            return StateEnum.SUSPENDED;
        } else {
            return StateEnum.ON;
        }
    }

    private static ServingStatusEnum externalServingStatus(Boolean bsRarelyLoaded) {
        return bsRarelyLoaded ? ServingStatusEnum.RARELY_SERVED : ServingStatusEnum.ELIGIBLE;
    }

    private static void itemWithExternalStatistics(KeywordGetItem keywordGetItem,
                                                   @Nullable Map<Long, StatValueAggregatorItem> showConditionStat) {
        keywordGetItem
                .withStatisticsSearch(
                        new Statistics().withClicks(0L).withImpressions(0L)
                )
                .withStatisticsNetwork(
                        new Statistics().withClicks(0L).withImpressions(0L)
                );
        if (showConditionStat != null) {
            if (showConditionStat.get(keywordGetItem.getId()) != null) {
                StatValueAggregatorItem stat =
                        showConditionStat.get(keywordGetItem.getId());
                keywordGetItem
                        .withStatisticsSearch(
                                new Statistics().withClicks(stat.getSearchClicks())
                                        .withImpressions(stat.getSearchShows())
                        )
                        .withStatisticsNetwork(
                                new Statistics().withClicks(stat.getNetworkClicks())
                                        .withImpressions(stat.getNetworkShows())
                        );
            }
        }
    }

    private Long priceToExternalBid(@Nullable BigDecimal price, Currency currency, AdGroupType adGroupType) {
        return ((price == null) || (price.compareTo(BigDecimal.ZERO) == 0)) ?
                getBidCurrencyMinPriceMicro(currency, adGroupType) :
                price.multiply(Money.MICRO_MULTIPLIER).longValue();
    }

    private Long getBidCurrencyMinPriceMicro(Currency currency, AdGroupType adGroupType) {
        BigDecimal defaultPrice =
                (adGroupType == AdGroupType.CPM_BANNER) ? currency.getMinCpmPrice() : currency.getMinPrice();
        return defaultPrice.multiply(Money.MICRO_MULTIPLIER).longValue();
    }
}
