package ru.yandex.direct.api.v5.entity.keywords.converter;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.yandex.direct.api.v5.keywords.KeywordUpdateItem;
import com.yandex.direct.api.v5.keywords.UpdateRequest;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ModelChangesHelper;
import ru.yandex.direct.api.v5.entity.keywords.container.UpdateInputItem;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.api.v5.common.RelevanceMatchCategoriesConverter.mergeAutotargetingCategoriesFromRequestWithDb;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class KeywordsUpdateRequestConverter {
    private final RelevanceMatchService relevanceMatchService;
    private final AdGroupService adGroupService;

    public KeywordsUpdateRequestConverter(RelevanceMatchService relevanceMatchService, AdGroupService adGroupService) {
        this.relevanceMatchService = relevanceMatchService;
        this.adGroupService = adGroupService;
    }

    public List<UpdateInputItem> convertRequest(UpdateRequest request, ClientId clientId) {
        Set<Long> requestIds = request.getKeywords().stream()
                .map(KeywordUpdateItem::getId)
                .filter(Objects::nonNull)
                .filter(id -> id > 0)
                .collect(Collectors.toSet());
        List<RelevanceMatch> clientRelevanceMatches = relevanceMatchService.getRelevanceMatchByIds(clientId,
                requestIds);
        List<Long> adGroupIds = mapList(clientRelevanceMatches, RelevanceMatch::getAdGroupId);
        List<AdGroup> adGroups = adGroupService.getAdGroups(clientId, adGroupIds);
        Map<Long, AdGroup> adGroupsMap = listToMap(adGroups, AdGroup::getId);
        return convertRequest(request, clientRelevanceMatches, adGroupsMap);
    }

    public List<UpdateInputItem> convertRequest(UpdateRequest request, List<RelevanceMatch> clientRelevanceMatches,
                                                Map<Long, AdGroup> adGroupsMap) {
        return mapList(request.getKeywords(), i -> convertItem(i, clientRelevanceMatches, adGroupsMap));
    }

    UpdateInputItem convertItem(KeywordUpdateItem item, List<RelevanceMatch> clientRelevanceMatches, Map<Long,
            AdGroup> adGroupsMap) {
        Map<Long, RelevanceMatch> clientRelevanceMatchesByIds = listToMap(clientRelevanceMatches,
                RelevanceMatch::getId);
        if (clientRelevanceMatchesByIds.containsKey(item.getId())) {
            RelevanceMatch relevanceMatch = clientRelevanceMatchesByIds.get(item.getId());
            return UpdateInputItem.createItemForRelevanceMatch(convertRelevanceMatch(item,
                    relevanceMatch, adGroupsMap.get(relevanceMatch.getAdGroupId()).getType()));
        } else {
            return UpdateInputItem.createItemForKeyword(convertKeywordItem(item));
        }
    }

    ModelChanges<Keyword> convertKeywordItem(KeywordUpdateItem item) {
        ModelChanges<Keyword> changes = new ModelChanges<>(item.getId(), Keyword.class);
        changes.processNotNull(item.getKeyword(), Keyword.PHRASE);
        ModelChangesHelper.processJaxbElement(changes, item.getUserParam1(), Keyword.HREF_PARAM1);
        ModelChangesHelper.processJaxbElement(changes, item.getUserParam2(), Keyword.HREF_PARAM2);
        return changes;
    }

    ModelChanges<RelevanceMatch> convertRelevanceMatch(KeywordUpdateItem item, RelevanceMatch relevanceMatch,
                                                       AdGroupType adGroupType) {
        ModelChanges<RelevanceMatch> changes = new ModelChanges<>(item.getId(), RelevanceMatch.class);
        ModelChangesHelper.processJaxbElement(changes, item.getUserParam1(), RelevanceMatch.HREF_PARAM1);
        ModelChangesHelper.processJaxbElement(changes, item.getUserParam2(), RelevanceMatch.HREF_PARAM2);
        Set<RelevanceMatchCategory> newRelevanceMatchCategories =
                mergeAutotargetingCategoriesFromRequestWithDb(item.getAutotargetingCategories(),
                        relevanceMatch.getRelevanceMatchCategories(), adGroupType);
        changes.process(newRelevanceMatchCategories, RelevanceMatch.RELEVANCE_MATCH_CATEGORIES);
        return changes;
    }
}
