package ru.yandex.direct.api.v5.entity.keywords.delegate;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywords.AddRequest;
import com.yandex.direct.api.v5.keywords.AddResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.keywords.KeywordsDefectTypes;
import ru.yandex.direct.api.v5.entity.keywords.KeywordsEndpoint;
import ru.yandex.direct.api.v5.entity.keywords.container.AddInputItem;
import ru.yandex.direct.api.v5.entity.keywords.converter.KeywordsAddRequestConverter;
import ru.yandex.direct.api.v5.entity.keywords.validation.KeywordsAddRequestValidator;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchAddOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.ConvertResultOperation;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.api.v5.entity.keywords.KeywordsEndpoint.KEYWORDS_CUSTOM_DEFECT_PRESENTATIONS;
import static ru.yandex.direct.result.ResultConverters.massResultValueConverter;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;

@Service
@ParametersAreNonnullByDefault
public class AddKeywordsDelegate
        extends OperationOnListDelegate<AddRequest, AddResponse, AddInputItem, Long> {

    private static final DefectPresentationsHolder KEYWORDS_ADD_CUSTOM_DEFECT_PRESENTATIONS =
            DefectPresentationsHolder.builderWithFallback(KEYWORDS_CUSTOM_DEFECT_PRESENTATIONS)
                    .register(CampaignDefectIds.Gen.CAMPAIGN_TYPE_NOT_SUPPORTED,
                            t -> KeywordsDefectTypes.keywordForbiddenInThisAdGroup())
                    .build();

    private static final PathConverter ADD_KEYWORDS_PATH_CONVERTER =
            MappingPathConverter.builder(KeywordsEndpoint.class, "capitalize")
                    .add(KeywordsEndpoint.getKeywordsCommonDict())
                    .add(Keyword.AD_GROUP_ID.name(), "AdGroupId")
                    .add(Keyword.PRICE.name(), "Bid")
                    .add(Keyword.PRICE_CONTEXT.name(), "ContextBid")
                    .add(Keyword.AUTOBUDGET_PRIORITY.name(), "StrategyPriority")
                    .add(RelevanceMatch.RELEVANCE_MATCH_CATEGORIES.name(), "AutotargetingCategories")
                    .build();
    private static final Logger logger = LoggerFactory.getLogger(AddKeywordsDelegate.class);

    private final KeywordsAddRequestValidator requestValidator;
    private final KeywordsAddRequestConverter requestConverter;
    private final ResultConverter resultConverter;
    private final ClientService clientService;
    private final KeywordService keywordService;
    private final RelevanceMatchService relevanceMatchService;
    private final RecentStatisticsService recentStatisticsService;

    @Autowired
    public AddKeywordsDelegate(
            ApiAuthenticationSource auth,
            KeywordsAddRequestValidator requestValidator,
            KeywordsAddRequestConverter requestConverter,
            ResultConverter resultConverter,
            ClientService clientService,
            KeywordService keywordService,
            RelevanceMatchService relevanceMatchService,
            RecentStatisticsService recentStatisticsService) {
        super(ADD_KEYWORDS_PATH_CONVERTER, auth);
        this.requestValidator = requestValidator;
        this.requestConverter = requestConverter;
        this.resultConverter = resultConverter;
        this.clientService = clientService;
        this.keywordService = keywordService;
        this.relevanceMatchService = relevanceMatchService;
        this.recentStatisticsService = recentStatisticsService;
    }

    @Override
    public ValidationResult<AddRequest, DefectType> validateRequest(AddRequest externalRequest) {
        logger.debug("validate request: {}", externalRequest);
        return requestValidator.validate(externalRequest);
    }

    @Override
    public List<AddInputItem> convertRequest(AddRequest externalRequest) {
        ApiUser targetUser = auth.getChiefSubclient();
        return requestConverter.convertRequest(externalRequest, targetUser.getClientId());
    }

    @Override
    @Nonnull
    public ValidationResult<List<AddInputItem>, DefectType> validateInternalRequest(
            List<AddInputItem> internalRequest) {
        return requestValidator.validateInternalRequest(internalRequest);
    }

    @Override
    public ApiMassResult<Long> processList(List<AddInputItem> internalRequest) {
        ApiUser operator = auth.getOperator();
        ApiUser targetUser = auth.getChiefSubclient();

        OperationCreator<AddInputItem, Operation<Long>> addKeywordCoreOperationCreator =
                inputItems -> new ConvertResultOperation<>(
                        keywordService.createKeywordAddOperation(targetUser.getClientId(),
                                operator.getUid(),
                                targetUser.getUid(),
                                mapList(inputItems, AddInputItem::getKeyword)),
                        massResultValueConverter(nullSafeConverter(AddedKeywordInfo::getId))
                );

        OperationCreator<AddInputItem, Operation<Long>> addRelevanceMatchCoreOperationCreator =
                inputItems -> new ConvertResultOperation<>(
                        createRelevanceMatchAddOperation(operator, targetUser,
                                mapList(inputItems, AddInputItem::getRelevanceMatch)),
                        massResultValueConverter(nullSafeConverter(AddedModelId::getId))
                );

        SplitAndMergeOperationAggregator<AddInputItem, Long> aggregatedAddOperation =
                SplitAndMergeOperationAggregator.builderForPartialOperations()
                        .addSubOperation(
                                AddInputItem::hasKeyword,
                                addKeywordCoreOperationCreator)
                        .addSubOperation(
                                AddInputItem::hasRelevanceMatch,
                                addRelevanceMatchCoreOperationCreator)
                        .build();
        MassResult<Long> aggregatedOperationResult = aggregatedAddOperation.execute(internalRequest);
        return resultConverter.toApiMassResult(aggregatedOperationResult, KEYWORDS_ADD_CUSTOM_DEFECT_PRESENTATIONS);
    }

    private RelevanceMatchAddOperation createRelevanceMatchAddOperation(User operator, User targetUser,
                                                                        List<RelevanceMatch> relevanceMatches) {
        Client client = checkNotNull(clientService.getClient(targetUser.getClientId()));

        ShowConditionAutoPriceParams autoPriceParams = new ShowConditionAutoPriceParams(
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null),
                recentStatisticsService
        );
        return relevanceMatchService.createPartialAddOperationWithAutoPrices(
                client.getWorkCurrency().getCurrency(), targetUser.getClientId(), operator.getUid(), relevanceMatches,
                autoPriceParams
        );
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new AddResponse().withAddResults(resultConverter.toActionResults(result, apiPathConverter));
    }
}
