package ru.yandex.direct.api.v5.entity.keywords.delegate;

import java.util.List;

import javax.annotation.Nullable;

import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.keywords.DeleteRequest;
import com.yandex.direct.api.v5.keywords.DeleteResponse;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.keywords.KeywordsDefectTypes;
import ru.yandex.direct.api.v5.entity.keywords.container.DeleteInputItem;
import ru.yandex.direct.api.v5.entity.keywords.converter.KeywordsDeleteRequestConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds;
import ru.yandex.direct.core.entity.keyword.service.KeywordDeleteOperation;
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchDeleteOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.keywords.KeywordsEndpoint.KEYWORDS_CUSTOM_DEFECT_PRESENTATIONS;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class DeleteKeywordsDelegate
        extends BaseApiServiceDelegate<DeleteRequest, DeleteResponse, List<DeleteInputItem>, ApiResult<Long>> {
    private static final DefectPresentationsHolder KEYWORDS_DELETE_CUSTOM_DEFECT_PRESENTATIONS =
            DefectPresentationsHolder.builderWithFallback(KEYWORDS_CUSTOM_DEFECT_PRESENTATIONS)
                    .register(CampaignDefectIds.Gen.CAMPAIGN_NO_WRITE_RIGHTS,
                            t -> KeywordsDefectTypes.keywordNotFound())
                    .build();
    public static final int MAX_IDS_PER_DELETE = 10_000;

    private final ResultConverter resultConverter;
    private final KeywordsDeleteRequestConverter keywordsDeleteRequestConverter;
    private final KeywordOperationFactory keywordOperationFactory;
    private final RelevanceMatchService relevanceMatchService;

    public DeleteKeywordsDelegate(
            ApiAuthenticationSource auth,
            ResultConverter resultConverter,
            KeywordsDeleteRequestConverter keywordsDeleteRequestConverter,
            KeywordOperationFactory keywordOperationFactory,
            RelevanceMatchService relevanceMatchService) {
        super(PathConverter.identity(), auth);
        this.resultConverter = resultConverter;
        this.keywordsDeleteRequestConverter = keywordsDeleteRequestConverter;
        this.keywordOperationFactory = keywordOperationFactory;
        this.relevanceMatchService = relevanceMatchService;
    }

    @Nullable
    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getSelectionCriteria(), DeleteRequest.PropInfo.SELECTION_CRITERIA.schemaName.getLocalPart())
                .item(request.getSelectionCriteria().getIds(), IdsCriteria.PropInfo.IDS.schemaName.getLocalPart())
                .check(eachNotNull())
                .check(Constraints.maxListSize(MAX_IDS_PER_DELETE),
                        KeywordsDefectTypes.maxElementsPerKeywordsDelete(MAX_IDS_PER_DELETE));
        return vb.getResult();
    }

    @Override
    public List<DeleteInputItem> convertRequest(DeleteRequest externalRequest) {
        ApiUser targetUser = auth.getChiefSubclient();
        return keywordsDeleteRequestConverter.convertRequest(externalRequest, targetUser.getClientId());
    }

    @Override
    public ApiResult<List<ApiResult<Long>>> processRequest(List<DeleteInputItem> internalRequest) {
        ApiUser operator = auth.getOperator();
        ApiUser targetUser = auth.getChiefSubclient();

        OperationCreator<DeleteInputItem, Operation<Long>> deleteKeywordCoreOperationCreator =
                inputItems -> createKeywordDeleteOperation(operator, targetUser,
                        FunctionalUtils.mapList(inputItems, DeleteInputItem::getHasKeywordId)
                );

        OperationCreator<DeleteInputItem, Operation<Long>> deleteRelevanceMatchCoreOperationCreator =
                inputItems -> createRelevanceMatchDeleteOperation(operator, targetUser,
                        FunctionalUtils.mapList(inputItems, DeleteInputItem::getRelevanceMatchId)
                );

        SplitAndMergeOperationAggregator<DeleteInputItem, Long> updateOperation =
                SplitAndMergeOperationAggregator.builderForPartialOperations()
                        .addSubOperation(
                                DeleteInputItem::hasKeywordId,
                                deleteKeywordCoreOperationCreator)
                        .addSubOperation(
                                DeleteInputItem::hasRelevanceMatchId,
                                deleteRelevanceMatchCoreOperationCreator)
                        .build();
        MassResult<Long> aggregateOperationResult = updateOperation.execute(internalRequest);
        return resultConverter.toApiMassResult(aggregateOperationResult, KEYWORDS_DELETE_CUSTOM_DEFECT_PRESENTATIONS);
    }

    private KeywordDeleteOperation createKeywordDeleteOperation(User operator, User targetUser,
                                                                List<Long> keywordIds) {
        return keywordOperationFactory.createKeywordsDeleteOperation(
                Applicability.PARTIAL, keywordIds, operator.getUid(),
                targetUser.getClientId());
    }

    private RelevanceMatchDeleteOperation createRelevanceMatchDeleteOperation(User operator, User targetUser,
                                                                              List<Long> relevanceMatchIds) {
        return relevanceMatchService.createDeleteOperation(
                targetUser.getClientId(), operator.getUid(), relevanceMatchIds,
                Applicability.PARTIAL);
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        List<ActionResult> updateResults = mapList(result.getResult(), r -> resultConverter.toActionResult(
                r, apiPathConverter, ActionResult::new, ActionResult::setId));
        return new DeleteResponse().withDeleteResults(updateResults);
    }
}
