package ru.yandex.direct.api.v5.entity.keywords.delegate;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywords.GetRequest;
import com.yandex.direct.api.v5.keywords.GetResponse;
import com.yandex.direct.api.v5.keywords.KeywordFieldEnum;
import com.yandex.direct.api.v5.keywords.KeywordGetItem;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.keywords.container.KeywordsGetContainer;
import ru.yandex.direct.api.v5.entity.keywords.converter.KeywordsGetRequestConverter;
import ru.yandex.direct.api.v5.entity.keywords.converter.KeywordsGetResponseConverter;
import ru.yandex.direct.api.v5.entity.keywords.validation.KeywordsGetRequestValidator;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.bids.container.ShowConditionSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.keywords.KeywordFieldEnum.PRODUCTIVITY;
import static com.yandex.direct.api.v5.keywords.KeywordFieldEnum.STATISTICS_NETWORK;
import static com.yandex.direct.api.v5.keywords.KeywordFieldEnum.STATISTICS_SEARCH;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class GetKeywordsDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, KeywordFieldEnum,
        ShowConditionSelectionCriteria, KeywordsGetContainer> {
    private static final Logger logger = LoggerFactory.getLogger(GetKeywordsDelegate.class);

    private final KeywordService keywordService;
    private final RelevanceMatchService relevanceMatchService;
    private final BidService bidService;
    private final EnumPropertyFilter<KeywordFieldEnum> propertyFilter;
    private final KeywordsGetResponseConverter keywordsGetResponseConverter;
    private final KeywordsGetRequestValidator requestValidator;
    private final KeywordsGetRequestConverter requestConverter;

    private static final PathConverter GET_KEYWORDS_PATH_CONVERTER =
            MappingPathConverter.builder(ResumeKeywordsDelegate.class, "capitalize")
                    .build();


    @Autowired
    public GetKeywordsDelegate(ApiAuthenticationSource auth,
                               KeywordService keywordService,
                               RelevanceMatchService relevanceMatchService,
                               BidService bidService,
                               PropertyFilter propertyFilter,
                               KeywordsGetResponseConverter keywordsGetResponseConverter,
                               KeywordsGetRequestValidator requestValidator,
                               KeywordsGetRequestConverter requestConverter) {
        super(GET_KEYWORDS_PATH_CONVERTER, auth);
        this.requestValidator = requestValidator;
        this.keywordService = keywordService;
        this.relevanceMatchService = relevanceMatchService;
        this.bidService = bidService;
        this.propertyFilter = EnumPropertyFilter.from(KeywordFieldEnum.class, propertyFilter);
        this.keywordsGetResponseConverter = keywordsGetResponseConverter;
        this.requestConverter = requestConverter;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        logger.debug("validate request {}", externalRequest);
        return requestValidator.validate(externalRequest);
    }

    @Override
    public List<KeywordsGetContainer> get(
            GenericGetRequest<KeywordFieldEnum, ShowConditionSelectionCriteria> getRequest) {
        logger.debug("get keywords");
        Long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();

        List<Bid> bids = bidService
                .getBids(clientId, operatorUid, getRequest.getSelectionCriteria(), getRequest.getLimitOffset(), true);
        List<Long> bidIds = mapList(bids, Bid::getId);
        List<Long> keywordIds = StreamEx.of(bids)
                .filter(bid -> bid.getType() == ShowConditionType.KEYWORD)
                .map(Bid::getId)
                .toList();
        List<Long> relevanceMatchBids = StreamEx.of(bids)
                .filter(bid -> bid.getType() == ShowConditionType.RELEVANCE_MATCH)
                .map(Bid::getId)
                .toList();

        Map<Long, KeywordsGetContainer> keywordsGetContainerMap = new HashMap<>();
        if (!keywordIds.isEmpty()) {
            List<Keyword> keywords =
                    keywordService.getKeywords(clientId, keywordIds);
            for (Keyword keyword : keywords) {
                keywordsGetContainerMap.put(keyword.getId(),
                        KeywordsGetContainer.createItemForKeyword(keyword));
            }
        }
        if (!relevanceMatchBids.isEmpty()) {
            List<RelevanceMatch> relevanceMatches =
                    relevanceMatchService.getRelevanceMatchByIds(clientId, relevanceMatchBids);
            for (RelevanceMatch relevanceMatch : relevanceMatches) {
                keywordsGetContainerMap.put(relevanceMatch.getId(),
                        KeywordsGetContainer.createItemForRelevanceMatch(relevanceMatch));
            }
        }
        return mapList(bidIds, keywordsGetContainerMap::get);
    }

    @Override
    public Set<KeywordFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public ShowConditionSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria selectionCriteria =
                externalRequest.getSelectionCriteria();
        return new ShowConditionSelectionCriteria()
                .withShowConditionIds(new HashSet<>(selectionCriteria.getIds()))
                .withAdGroupIds(new HashSet<>(selectionCriteria.getAdGroupIds()))
                .withCampaignIds(new HashSet<>(selectionCriteria.getCampaignIds()))
                .withStates(requestConverter.convertStates(selectionCriteria.getStates()))
                .withStatuses(requestConverter.convertStatuses(selectionCriteria.getStatuses()))
                .withServingStatuses(requestConverter.convertServingStatuses(selectionCriteria.getServingStatuses()))
                .withModifiedSince(requestConverter.convertModifiedSince(selectionCriteria.getModifiedSince()));

    }

    /**
     * Корректируем operationSummary (выставляем specialProcessingObjectCost true),
     * т.к. требуется списание 3 баллов за каждые 2000 фраз,
     * если запрошен хотя бы один из параметров Productivity, StatisticsSearch, StatisticsNetwork.
     * <p>
     * 1 балл за каждые 2000 фраз — в противном случае.
     */
    @Override
    public OperationSummary correctOperationSummary(
            GenericGetRequest<KeywordFieldEnum, ShowConditionSelectionCriteria> getRequest,
            ApiResult<List<KeywordsGetContainer>> apiResult) {
        if (!apiResult.isSuccessful()) {
            return apiResult.getOperationSummary();
        }
        if (getRequest.getRequestedFields().contains(STATISTICS_NETWORK)
                || getRequest.getRequestedFields().contains(STATISTICS_SEARCH)
                || getRequest.getRequestedFields().contains(PRODUCTIVITY)) {
            return OperationSummary.successful(apiResult.getSuccessfulObjectsCount(), true);
        } else {
            return apiResult.getOperationSummary();
        }
    }

    @Override
    public GetResponse convertGetResponse(List<KeywordsGetContainer> keywordGetContainers,
                                          Set<KeywordFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        logger.debug("convert response - result {}, requestedFields {}, limitedBy {}", keywordGetContainers,
                requestedFields,
                limitedBy);
        Boolean requestedStat =
                requestedFields.contains(STATISTICS_NETWORK) || requestedFields.contains(STATISTICS_SEARCH);
        ClientId clientId = auth.getChiefSubclient().getClientId();
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);
        if (!keywordGetContainers.isEmpty()) {
            List<KeywordGetItem> getItems =
                    keywordsGetResponseConverter.convert(keywordGetContainers, clientId, requestedStat);
            propertyFilter.filterProperties(getItems, requestedFields);
            response.withKeywords(getItems);
        }
        return response;
    }
}
