package ru.yandex.direct.api.v5.entity.keywords.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.entity.keywords.container.KeywordsGetContainer;
import ru.yandex.direct.api.v5.entity.keywords.container.StatValueAggregatorItem;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.StatValueAggregator;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

@Service
public class StatisticService {
    private final RecentStatisticsService recentStatisticsService;
    private static final int DEFAULT_STATISTIC_PERIOD = 28;

    @Autowired
    public StatisticService(RecentStatisticsService recentStatisticsService) {
        this.recentStatisticsService = recentStatisticsService;
    }

    /**
     * Для указанных фраз возвращает статистику за 28 календарных дней.
     * <p>
     * Должна совпадать со статистикой web-интерфейса (подробнее в тикете <a href="https://st.yandex-team.ru/DIRECT-89316">DIRECT-89316</a>)
     */
    public Map<Long, StatValueAggregatorItem> getPhraseStatistics(List<KeywordsGetContainer> showConditions) {
        List<PhraseStatisticsRequest> requests =
                showConditions.stream()
                        .map(showCondition -> new PhraseStatisticsRequest.Builder()
                                .withPhraseId(showCondition.getId())
                                .withAdGroupId(showCondition.getAdGroupId())
                                .withCampaignId(showCondition.getCampaignId())
                                .build())
                        .collect(Collectors.toList());
        Map<PhraseStatisticsResponse.PhraseStatIndex, StatValueAggregator> phraseStatistics =
                recentStatisticsService.getPhraseStatistics(requests, getDateRange());

        Map<Long, List<StatValueAggregatorItem>> statisticsMap = EntryStream.of(phraseStatistics)
                .mapKeys(PhraseStatisticsResponse.PhraseStatIndex::getPhraseId)
                .mapValues(value -> new StatValueAggregatorItem()
                        .withSearchClicks(value.getSearchClicks())
                        .withSearchShows(value.getSearchShows())
                        .withNetworkClicks(value.getNetworkClicks())
                        .withNetworkShows(value.getNetworkShows()))
                .grouping();

        return EntryStream.of(statisticsMap)
                .mapValues(items -> items.stream().reduce((v1, v2) -> new StatValueAggregatorItem()
                        .withSearchShows(v1.getSearchShows() + v2.getSearchShows())
                        .withSearchClicks(v1.getSearchClicks() + v2.getSearchClicks())
                        .withNetworkShows(v1.getNetworkShows() + v2.getNetworkShows())
                        .withNetworkClicks(v1.getNetworkClicks() + v2.getNetworkClicks())))
                .filterValues(Optional::isPresent)
                .flatMapValues(StreamEx::of)
                .toMap();

    }

    private DateRange getDateRange() {
        return new DateRange()
                .withFromExclusive(LocalDate.now().minusDays(DEFAULT_STATISTIC_PERIOD))
                .withToInclusive(LocalDate.now());
    }
}
