package ru.yandex.direct.api.v5.entity.keywords.validation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywords.AddRequest;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.keywords.container.AddInputItem;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.priorityWontBeAcceptedNotAutoBudgetStrategy;
import static ru.yandex.direct.api.v5.entity.keywords.KeywordsDefectTypes.maxElementsPerKeywordsAdd;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

@Component
@ParametersAreNonnullByDefault
public class KeywordsAddRequestValidator {

    static final int MAX_ELEMENTS_PER_ADD = 1_000;

    public ValidationResult<AddRequest, DefectType> validate(AddRequest externalRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getKeywords(), AddRequest.PropInfo.KEYWORDS.schemaName.getLocalPart())
                .check(maxListSize(MAX_ELEMENTS_PER_ADD), maxElementsPerKeywordsAdd(MAX_ELEMENTS_PER_ADD));
        return vb.getResult();
    }

    public ValidationResult<List<AddInputItem>, DefectType> validateInternalRequest(
            List<AddInputItem> internalRequest) {
        return ListValidationBuilder.<AddInputItem, DefectType>of(internalRequest)
                .checkEachBy(KeywordsAddRequestValidator::validatePriority)
                .checkEach(notEmptyRelevanceMatchCategories())
                .getResult();
    }

    private static ValidationResult<AddInputItem, DefectType> validatePriority(AddInputItem inputItem) {
        final ValidationResult<AddInputItem, DefectType> rootVr = new ValidationResult<>(inputItem);
        if (inputItem.getPriorityIsSetForManualStrategy()) {
            rootVr.getOrCreateSubValidationResult(
                    PathHelper.field(RelevanceMatch.AUTOBUDGET_PRIORITY.name()), null)
                    .addWarning(priorityWontBeAcceptedNotAutoBudgetStrategy());
        }
        return rootVr;
    }

    private static Constraint<AddInputItem, DefectType> notEmptyRelevanceMatchCategories() {
        return Constraint.fromPredicate(item -> !item.hasRelevanceMatch()
                        || item.getRelevanceMatch().getRelevanceMatchCategories() == null
                        || !isEmpty(item.getRelevanceMatch().getRelevanceMatchCategories()),
                DefectTypes.disableAllAutotargetingCategoriesIsForbidden());
    }
}
