package ru.yandex.direct.api.v5.entity.keywords.validation;

import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.keywords.GetRequest;
import com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.keywords.GetRequest.PropInfo.SELECTION_CRITERIA;
import static com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria.PropInfo.AD_GROUP_IDS;
import static com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria.PropInfo.CAMPAIGN_IDS;
import static com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria.PropInfo.IDS;
import static com.yandex.direct.api.v5.keywords.KeywordsSelectionCriteria.PropInfo.MODIFIED_SINCE;
import static java.util.Arrays.asList;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypeCreators.invalidRequestParamsIncorrectDate;
import static ru.yandex.direct.api.v5.validation.DefectTypeCreators.invalidRequestParamsInvalidDateFormat;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.validation.Predicates.validDateTime;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class KeywordsGetRequestValidator {
    static final int MAX_ADGROUP_IDS_COUNT = 1000;
    public static final int MAX_CAMPAIGN_IDS_COUNT = 10;
    private static final String MODIFIED_SINCE_DATE_PATTERN = "^\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}Z$";


    public ValidationResult<GetRequest, DefectType> validate(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getSelectionCriteria(), SELECTION_CRITERIA.schemaName.getLocalPart())
                .check(notNull())
                .checkBy(this::validateSelectionCriteria, When.isValid());

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes, When.notNull());
        return vb.getResult();
    }

    private ValidationResult<KeywordsSelectionCriteria, DefectType> validateSelectionCriteria(
            KeywordsSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<KeywordsSelectionCriteria, DefectType> vb = ItemValidationBuilder.of(selectionCriteria);
        vb.check(fromPredicate(
                sc -> !sc.getIds().isEmpty() || !sc.getAdGroupIds().isEmpty() || !sc.getCampaignIds().isEmpty(),
                missedParamsInSelection(String.join(
                        ", ", asList(IDS.schemaName.getLocalPart(), AD_GROUP_IDS.schemaName.getLocalPart(),
                                CAMPAIGN_IDS.schemaName.getLocalPart())))));
        vb.item(selectionCriteria.getIds(), IDS.schemaName.getLocalPart())
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());
        vb.item(selectionCriteria.getAdGroupIds(), AD_GROUP_IDS.schemaName.getLocalPart())
                .check(maxListSize(MAX_ADGROUP_IDS_COUNT), maxIdsInSelection());
        vb.item(selectionCriteria.getCampaignIds(), CAMPAIGN_IDS.schemaName.getLocalPart())
                .check(maxListSize(MAX_CAMPAIGN_IDS_COUNT), maxIdsInSelection());
        vb.item(selectionCriteria.getModifiedSince(), MODIFIED_SINCE.schemaName.getLocalPart())
                .check(Constraints.matchPattern(MODIFIED_SINCE_DATE_PATTERN),
                        invalidRequestParamsInvalidDateFormat(MODIFIED_SINCE.schemaName.getLocalPart()))
                .check(Constraint.fromPredicate(validDateTime(DateTimeFormatter.ISO_OFFSET_DATE_TIME),
                        invalidRequestParamsIncorrectDate(MODIFIED_SINCE.schemaName.getLocalPart())), When.isValid());

        return vb.getResult();
    }
}
