package ru.yandex.direct.api.v5.entity.keywordsresearch.delegate;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateErrorItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateOperationEnum;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateRequest;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateRequestItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponse;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseAddItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseUpdateItem;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnRequestWithListDelegate;
import ru.yandex.direct.api.v5.entity.keywordsresearch.DeduplicateDefectTypes;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.result.ApiResultState;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.keyword.processing.KeywordNormalizer;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedKeyword;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedKeywordWithMinuses;
import ru.yandex.direct.core.entity.keyword.processing.NormalizedWord;
import ru.yandex.direct.core.entity.keyword.processing.ProcessedKeyword;
import ru.yandex.direct.core.entity.keyword.processing.glue.KeywordGluer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.KeywordUngluer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueContainer;
import ru.yandex.direct.core.entity.keyword.processing.unglue.UnglueResult;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseDefectIds;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseValidator;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.model.KeywordWithMinuses;
import ru.yandex.direct.libs.keywordutils.model.SingleKeyword;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static one.util.streamex.MoreCollectors.countingInt;
import static ru.yandex.direct.api.v5.entity.keywordsresearch.Constants.MAX_KEYWORDS_PER_DEDUPLICATE_REQUEST;
import static ru.yandex.direct.api.v5.validation.DefectTypes.duplicatedElement;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notEmptyCollection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notLessThan;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.validId;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicates;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator.keywordSyntaxValidator;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class DeduplicateDelegate extends OperationOnRequestWithListDelegate<DeduplicateRequest,
        DeduplicateResponse,
        DeduplicateKeywordItem,
        DeduplicateKeywordsRequest,
        DeduplicateKeywordResponse> {
    private static final DefectPresentationsHolder DEDUPLICATE_CUSTOM_DEFECT_PRESENTATIONS =
            DefectPresentationsHolder.builder()
                    .register(PhraseDefectIds.Gen.NO_PLUS_WORDS,
                            DeduplicateDefectTypes.cannotContainsOnlyMinusWords())
                    .register(PhraseDefectIds.String.NOT_SINGLE_MINUS_WORD,
                            DeduplicateDefectTypes.noMinusPhrasesOnlyWords())
                    .register(PhraseDefectIds.String.MINUS_WORD_DELETE_PLUS_WORD,
                            t -> DeduplicateDefectTypes.minusWordsCannotSubtractPlusWords(t.getAllInvalidSubstrings()))
                    .register(PhraseDefectIds.String.ILLEGAL_CHARACTERS, DeduplicateDefectTypes.invalidChars())
                    .register(PhraseDefectIds.Gen.INVALID_PLUS_MARK, DeduplicateDefectTypes.incorrectUseOfPlusSign())
                    .register(PhraseDefectIds.Gen.INVALID_EXCLAMATION_MARK,
                            DeduplicateDefectTypes.incorrectUseOfExclamationSign())
                    .register(PhraseDefectIds.Gen.INVALID_MINUS_MARK, DeduplicateDefectTypes.incorrectUseOfMinusSign())
                    .register(PhraseDefectIds.Gen.MINUS_WORD_INSIDE_BRACKETS_OR_QUOTES,
                            DeduplicateDefectTypes.modifiersInsideSquareBrackets())
                    .register(PhraseDefectIds.Gen.INVALID_BRACKETS, DeduplicateDefectTypes.unpairedSquareBrackets())
                    .register(PhraseDefectIds.String.TOO_MANY_WORDS,
                            t -> DeduplicateDefectTypes.tooManyWords(t.getMaxWords()))
                    .register(PhraseDefectIds.String.TOO_LONG_WORD,
                            t -> DeduplicateDefectTypes.tooLongKeyword(t.getMaxLength()))
                    .register(PhraseDefectIds.Gen.ONLY_STOP_WORDS, DeduplicateDefectTypes.containsOnlyStopWords())
                    .register(PhraseDefectIds.Gen.INVALID_QUOTES, DeduplicateDefectTypes.unpairedQuotes())
                    .register(PhraseDefectIds.Gen.INVALID_POINT, DeduplicateDefectTypes.invalidUseOfDot())
                    .register(PhraseDefectIds.Gen.BOTH_QUOTES_AND_MINUS_WORDS,
                            DeduplicateDefectTypes.bothQuotesAndMinusWords())
                    .build();

    private final ResultConverter resultConverter;
    private final KeywordNormalizer normalizer;
    private final KeywordUngluer ungluer;
    private final StopWordService stopWordService;
    private final KeywordWithLemmasFactory keywordFactory;
    private final SingleKeywordsCache singleKeywordsCache;

    @Autowired
    DeduplicateDelegate(ApiAuthenticationSource auth,
                        ResultConverter resultConverter,
                        KeywordNormalizer normalizer,
                        KeywordUngluer ungluer,
                        StopWordService stopWordService,
                        KeywordWithLemmasFactory keywordFactory,
                        SingleKeywordsCache singleKeywordsCache) {
        super(ApiPathConverter.forKeywordsResearch(), auth);

        this.resultConverter = resultConverter;
        this.normalizer = normalizer;
        this.ungluer = ungluer;
        this.stopWordService = stopWordService;
        this.keywordFactory = keywordFactory;
        this.singleKeywordsCache = singleKeywordsCache;
    }

    @Nullable
    @Override
    public ValidationResult<DeduplicateRequest, DefectType> validateRequest(DeduplicateRequest request) {
        ItemValidationBuilder<DeduplicateRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getKeywords(), "Keywords")
                .check(notEmptyCollection())
                .check(eachNotNull())
                .check(maxListSize(MAX_KEYWORDS_PER_DEDUPLICATE_REQUEST),
                        maxElementsPerRequest(MAX_KEYWORDS_PER_DEDUPLICATE_REQUEST), When.isValid());
        return vb.getResult();
    }

    @Override
    public DeduplicateKeywordsRequest convertRequest(DeduplicateRequest request) {
        List<DeduplicateKeywordItem> deduplicateKeywordItems =
                mapList(request.getKeywords(), DeduplicateKeywordItem::new);
        return new DeduplicateKeywordsRequest(request.getOperation(), deduplicateKeywordItems);
    }

    @Nonnull
    @Override
    public ValidationResult<List<DeduplicateKeywordItem>, DefectType> validateInternalElements(
            List<DeduplicateKeywordItem> internalRequest) {
        Map<Long, Integer> idCounts = StreamEx.of(internalRequest)
                .map(DeduplicateKeywordItem::getId)
                .filter(Objects::nonNull)
                .collect(groupingBy(identity(), countingInt()));
        ListValidationBuilder<DeduplicateKeywordItem, DefectType> vb = ListValidationBuilder.of(internalRequest);
        vb.checkEachBy(itemValidator(idCounts));
        vb.checkEachBy(keywordValidator());
        return vb.getResult();
    }

    @Override
    public ApiMassResult<DeduplicateKeywordResponse> processRequestWithList(
            DeduplicateKeywordsRequest requestWithValidItems) {
        List<DeduplicateKeywordItem> validItems = requestWithValidItems.getItems();
        Map<SingleKeyword, List<NormalizedWord<SingleKeyword>>> normalizationMinusWordMap = new HashMap<>();
        Map<Keyword, NormalizedKeyword> normalizationKeywordMap = new HashMap<>();
        List<NormalizedKeywordWithMinuses> normalizedKeywords = mapList(validItems,
                t -> normalizer.normalizeKeywordWithMinuses(t.getParsedKeyword(),
                        normalizationMinusWordMap,
                        normalizationKeywordMap));

        List<NormalizedKeywordWithMinuses> keywordsToProcess = normalizedKeywords;

        if (needToPerformOperation(requestWithValidItems.getOperations(),
                DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)) {
            keywordsToProcess = unglueKeywords(keywordsToProcess);
        }
        // склеиваем дублирующиеся минус-слова в разных формах, например -база -базой
        keywordsToProcess = StreamEx.of(keywordsToProcess).map(this::deduplicateMinusWords).toList();
        if (needToPerformOperation(requestWithValidItems.getOperations(),
                DeduplicateOperationEnum.MERGE_DUPLICATES)) {
            keywordsToProcess = KeywordGluer.glueKeywords(singleKeywordsCache, keywordsToProcess);
        }

        // convert to results
        List<ApiResult<DeduplicateKeywordResponse>> itemResponses =
                constructInternalResponse(validItems, normalizedKeywords, keywordsToProcess);

        return new ApiMassResult<>(itemResponses, emptyList(), emptyList(), ApiResultState.SUCCESSFUL);
    }

    private List<ApiResult<DeduplicateKeywordResponse>> constructInternalResponse(
            List<DeduplicateKeywordItem> validItems,
            List<NormalizedKeywordWithMinuses> normalizedKeywords,
            List<NormalizedKeywordWithMinuses> processedKeywords) {

        return EntryStream.of(validItems)
                .mapKeyValue((index, item) -> toApiResult(item,
                        constructProcessedKeyword(normalizedKeywords.get(index), processedKeywords.get(index)),
                        // если ключевые фразы до и после склейки не совпадают,
                        // значит она была склеена с другой фразой, а потому является дубликатом
                        !normalizedKeywords.get(index).getKeyword().equals(processedKeywords.get(index).getKeyword())))
                .toList();
    }

    /**
     * исходная фраза, исходные минус-слова -> обработанная фраза, обработанные минус-слова в исходном виде
     */
    private ProcessedKeyword constructProcessedKeyword(NormalizedKeywordWithMinuses normalizedKeyword,
                                                       NormalizedKeywordWithMinuses processedKeyword) {
        return new ProcessedKeyword(
                new KeywordWithMinuses(normalizedKeyword.getKeyword().getOriginal(),
                        mapList(normalizedKeyword.getOriginalMinusWords(),
                                mw -> new Keyword(singletonList(mw)))),
                new KeywordWithMinuses(processedKeyword.getKeyword().getNormalized(),
                        mapList(processedKeyword.getOriginalMinusWords(),
                                mw -> new Keyword(singletonList(mw)))));
    }

    /**
     * Расклеивает ключевые фразы.
     *
     * @param keywords исходные ключевые фразы
     * @return расклеенные ключевые фразы
     */
    private List<NormalizedKeywordWithMinuses> unglueKeywords(List<NormalizedKeywordWithMinuses> keywords) {
        List<UnglueContainer> unglueSource = IntStream.range(0, keywords.size())
                .mapToObj(i -> new UnglueContainer(i, 1, keywords.get(i)))
                .collect(toList());
        Map<Integer, UnglueResult> unglueResults =
                Maps.uniqueIndex(ungluer.unglue(unglueSource, emptyList()), UnglueResult::getIndex);

        List<NormalizedKeywordWithMinuses> ungluedKeywords = new ArrayList<>(keywords.size());

        for (int i = 0; i < keywords.size(); i++) {
            if (unglueResults.containsKey(i)) {
                ungluedKeywords.add(i, convertToUnglued(keywords.get(i), unglueResults.get(i)));
            } else {
                ungluedKeywords.add(keywords.get(i));
            }
        }
        return ungluedKeywords;
    }

    /**
     * Показывает необходимо ли выполнить заданную операцию в текущем запросе.
     *
     * @param requestOperations операции из запроса, если список пустой - заданная операция выполняется.
     * @param operation заданная операция:
     * MERGE_DUPLICATES - выполнить склейку фраз,
     * ELIMINATE_OVERLAPPING - выполнить расклейку (кросс-минусовку) фраз
     * @return true - если заданную операцию необходимо выполнить, иначе false
     */
    private boolean needToPerformOperation(List<DeduplicateOperationEnum> requestOperations,
                                           DeduplicateOperationEnum operation) {
        return requestOperations.isEmpty() || requestOperations.contains(operation);
    }

    @Override
    public DeduplicateResponse convertResponse(ApiResult<List<ApiResult<DeduplicateKeywordResponse>>> result) {
        List<Long> toDelete = new ArrayList<>();

        List<ApiResult<DeduplicateKeywordResponse>> results = result.getResult();
        List<DeduplicateErrorItem> errors = collectErrors(results);

        Map<KeywordWithMinuses, ResponseItem> responses = new HashMap<>();

        StreamEx<DeduplicateKeywordResponse> toRecommend = StreamEx.of(results)
                .filter(ApiResult::isSuccessful)
                .map(ApiResult::getResult)
                .reverseSorted(Comparator.comparingDouble(item1 -> nvl(item1.getWeight(), 0L)));

        for (DeduplicateKeywordResponse keywordResponse : toRecommend) {
            if (keywordResponse.isDuplicate()) {
                if (keywordResponse.getId() != null) {
                    toDelete.add(keywordResponse.getId());
                }
                continue;
            }
            KeywordWithMinuses originalKeywordsWithOriginalMinusWords = keywordResponse.getKeyword().getBefore();
            KeywordWithMinuses normalizedKeywordsWithProcessedMinusWords = keywordResponse.getKeyword().getAfter();
            KeywordWithMinuses originalKeywordsWithProcessedMinusWords =
                    new KeywordWithMinuses(originalKeywordsWithOriginalMinusWords.getKeyword(),
                            normalizedKeywordsWithProcessedMinusWords.getMinusKeywords());

            ResponseItem item = responses.computeIfAbsent(normalizedKeywordsWithProcessedMinusWords,
                    kw -> new ResponseItem());
            item.setKeyword(originalKeywordsWithProcessedMinusWords);
            if (keywordResponse.getId() != null) {
                switch (item.getOperation()) {
                    case ADD:
                        // если фразы до и после обновления совпадают, то не обновляем
                        if (item.getKeyword().equals(originalKeywordsWithOriginalMinusWords)) {
                            item.setOperation(Operation.IGNORE);
                            item.setId(keywordResponse.getId());
                        } else if (isValid(originalKeywordsWithProcessedMinusWords)) {
                            item.setOperation(Operation.UPDATE);
                            item.setId(keywordResponse.getId());
                        } else {
                            // после склейки ключевых фраз и приведения обратно к оригиналам эта фраза стала невалидной,
                            // удаляем, у нее есть пара (или больше), одна из которых валидна
                            toDelete.add(keywordResponse.getId());
                        }
                        break;

                    case UPDATE:
                    case IGNORE:
                        // такой update уже был, это слово дублируется, удаляем
                        toDelete.add(keywordResponse.getId());
                        break;
                }
            }
        }

        return createResponse(responses.values(), toDelete, errors);
    }

    private boolean isValid(KeywordWithMinuses keyword) {
        Set<String> keywords = StreamEx.of(keyword.getKeyword().getAllKeywords()).map(Objects::toString).toSet();
        Set<String> minusWords = StreamEx.of(keyword.getMinusKeywords()).map(Objects::toString).toSet();
        return Sets.intersection(keywords, minusWords).isEmpty();
    }

    private ListItemValidator<DeduplicateKeywordItem, DefectType> itemValidator(Map<Long, Integer> idsCount) {
        return (index, item) -> {
            ItemValidationBuilder<DeduplicateKeywordItem, DefectType> vb =
                    ItemValidationBuilder.of(item);
            vb.item(item.getId(), DeduplicateRequestItem.PropInfo.ID.propertyName)
                    .check(validId())
                    .check(id -> idsCount.get(id) == 1 ? null : duplicatedElement(), When.isValidAnd(When.notNull()));
            vb.item(item.getWeight(), DeduplicateRequestItem.PropInfo.WEIGHT.propertyName)
                    .check(notLessThan(0L));
            return vb.getResult();
        };
    }

    private ListItemValidator<DeduplicateKeywordItem, DefectType> keywordValidator() {
        return (index, item) -> {
            ItemValidationBuilder<DeduplicateKeywordItem, Defect> vb =
                    ItemValidationBuilder.of(item);
            vb.item(item.getKeyword(), DeduplicateRequestItem.PropInfo.KEYWORD.propertyName)
                    .checkBy(keywordSyntaxValidator())
                    .checkBy(keyword -> new PhraseValidator(stopWordService, keywordFactory, item.getParsedKeyword())
                                    .apply(keyword),
                            When.isValid());
            return resultConverter.convertValidationResult(vb.getResult(), DEDUPLICATE_CUSTOM_DEFECT_PRESENTATIONS);
        };
    }

    private NormalizedKeywordWithMinuses deduplicateMinusWords(NormalizedKeywordWithMinuses keyword) {
        Map<SingleKeyword, NormalizedWord<SingleKeyword>> originalToNormalizedMinusWords =
                StreamEx.of(keyword.getMinusWords())
                        .toMap(NormalizedWord::getOriginalWord, identity());
        List<Keyword> originalMinusWords = StreamEx.of(keyword.getMinusWords())
                .map(NormalizedWord::getOriginalWord)
                .map(skw -> new Keyword(List.of(skw)))
                .toList();
        var deduplicated = removeDuplicates(singleKeywordsCache, originalMinusWords);
        var finalMinusWords = StreamEx.of(deduplicated)
                .map(kw -> kw.getAllKeywords().get(0))
                .map(originalToNormalizedMinusWords::get)
                .toList();
        return new NormalizedKeywordWithMinuses(keyword.getKeyword(), finalMinusWords);
    }

    private NormalizedKeywordWithMinuses convertToUnglued(NormalizedKeywordWithMinuses source,
                                                          UnglueResult unglueResult) {
        return source.appendMinuses(unglueResult.getAddedMinusWords());
    }

    private ApiResult<DeduplicateKeywordResponse> toApiResult(DeduplicateKeywordItem request,
                                                              ProcessedKeyword processed,
                                                              boolean isDuplicate) {
        return ApiResult.successful(new DeduplicateKeywordResponse(request.getId(),
                processed, request.getWeight(), isDuplicate));
    }

    private List<DeduplicateErrorItem> collectErrors(List<ApiResult<DeduplicateKeywordResponse>> results) {
        return IntStream.range(0, results.size())
                .mapToObj(index -> {
                    ApiResult<DeduplicateKeywordResponse> result = results.get(index);
                    if (result.isSuccessful()) {
                        return null;
                    }

                    DeduplicateErrorItem error = resultConverter.toActionResult(result,
                            apiPathConverter,
                            DeduplicateErrorItem::new,
                            null);
                    error.setPosition(index + 1L); // position is 1-based
                    return error;
                })
                .filter(Objects::nonNull)
                .collect(toList());
    }

    private static DeduplicateResponse createResponse(Collection<ResponseItem> responses,
                                                      List<Long> toDelete,
                                                      List<DeduplicateErrorItem> errors) {
        List<DeduplicateResponseAddItem> toAdd = new ArrayList<>();
        List<DeduplicateResponseUpdateItem> toUpdate = new ArrayList<>();

        for (ResponseItem item : responses) {
            switch (item.getOperation()) {
                case ADD:
                    toAdd.add(new DeduplicateResponseAddItem()
                            .withKeyword(item.getKeyword().toString()));
                    break;
                case UPDATE:
                    toUpdate.add(new DeduplicateResponseUpdateItem()
                            .withId(item.getId())
                            .withKeyword(item.getKeyword().toString()));
                    break;
                case IGNORE:
                    // do nothing
                    break;
            }
        }

        DeduplicateResponse response = new DeduplicateResponse();
        if (!toAdd.isEmpty()) {
            response.withAdd(toAdd);
        }

        if (!toUpdate.isEmpty()) {
            response.withUpdate(toUpdate);
        }

        if (!toDelete.isEmpty()) {
            response.withDelete(new IdsCriteria().withIds(toDelete));
        }

        if (!errors.isEmpty()) {
            response.withFailure(errors);
        }

        return response;
    }


    private static class ResponseItem {
        private Long id;
        private KeywordWithMinuses keyword;
        private Operation operation;
        private Long weight;

        private ResponseItem() {
            this.operation = Operation.ADD;
        }

        public Long getId() {
            return id;
        }

        public void setId(Long id) {
            this.id = id;
        }

        public KeywordWithMinuses getKeyword() {
            return keyword;
        }

        public void setKeyword(KeywordWithMinuses keyword) {
            this.keyword = keyword;
        }

        public Operation getOperation() {
            return operation;
        }

        public void setOperation(Operation operation) {
            this.operation = operation;
        }

        public Long getWeight() {
            return weight;
        }

        public void setWeight(Long weight) {
            this.weight = weight;
        }
    }

    private enum Operation {
        ADD,
        UPDATE,
        IGNORE
    }
}
