package ru.yandex.direct.api.v5.entity.keywordsresearch.delegate;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum;
import com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeItem;
import com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeRequest;
import com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.AdvqClient;
import ru.yandex.direct.advq.CheckMinHitsResult;
import ru.yandex.direct.advq.checkminhits.CheckMinHitsItem;
import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.validation.DefectPresentationsHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.keywordsresearch.AdvqDefectTypes;
import ru.yandex.direct.api.v5.entity.keywordsresearch.service.HasSearchVolumeInnerRequest;
import ru.yandex.direct.api.v5.entity.keywordsresearch.service.KeywordSearchVolumes;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseDefectIds;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseSyntaxValidator;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseValidator;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.utils.RuntimeServiceException;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.ALL_DEVICES;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.advq.AdvqClient.DEVICES_ALL;
import static ru.yandex.direct.advq.AdvqClient.DEVICES_DESKTOP;
import static ru.yandex.direct.advq.AdvqClient.DEVICES_PHONE;
import static ru.yandex.direct.advq.AdvqClient.DEVICES_TABLET;
import static ru.yandex.direct.api.v5.validation.DefectTypes.absent;
import static ru.yandex.direct.api.v5.validation.DefectTypes.duplicatedElement;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.DefectTypes.requiredButEmpty;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.minListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.unique;
import static ru.yandex.direct.libs.keywordutils.parser.KeywordParser.parseWithMinuses;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class HasSearchVolumeDelegate extends BaseApiServiceDelegate<
        HasSearchVolumeRequest, HasSearchVolumeResponse,
        HasSearchVolumeInnerRequest, KeywordSearchVolumes> {
    private static final DefectPresentationsHolder HAS_SEARCH_VOLUME_CUSTOM_DEFECT_PRESENTATIONS =
            DefectPresentationsHolder.builder()
                    .register(PhraseDefectIds.Gen.NO_PLUS_WORDS, AdvqDefectTypes.cannotContainsOnlyMinusWords())
                    .register(PhraseDefectIds.String.NOT_SINGLE_MINUS_WORD, AdvqDefectTypes.noMinusPhrasesOnlyWords())
                    .register(PhraseDefectIds.String.MINUS_WORD_DELETE_PLUS_WORD,
                            AdvqDefectTypes.minusWordsCannotSubtractPlusWords())
                    .register(PhraseDefectIds.String.ILLEGAL_CHARACTERS, AdvqDefectTypes.invalidChars())
                    .register(PhraseDefectIds.Gen.INVALID_MINUS_MARK, AdvqDefectTypes.incorrectUseOfMinusSign())
                    .register(PhraseDefectIds.Gen.MINUS_WORD_INSIDE_BRACKETS_OR_QUOTES,
                            AdvqDefectTypes.modifiersInsideSquareBrackets())
                    .register(PhraseDefectIds.Gen.INVALID_BRACKETS, AdvqDefectTypes.unpairedSquareBrackets())
                    .register(PhraseDefectIds.String.TOO_MANY_WORDS, t -> AdvqDefectTypes.tooManyWords(t.getMaxWords()))
                    .register(PhraseDefectIds.String.TOO_LONG_WORD,
                            t -> AdvqDefectTypes.tooLongKeyword(t.getMaxLength()))
                    .register(PhraseDefectIds.Gen.ONLY_STOP_WORDS, AdvqDefectTypes.containsOnlyStopWords())
                    .register(PhraseDefectIds.Gen.INVALID_QUOTES, AdvqDefectTypes.unpairedQuotes())
                    .register(PhraseDefectIds.Gen.INVALID_POINT, AdvqDefectTypes.invalidUseOfDot())
                    .register(PhraseDefectIds.Gen.BOTH_QUOTES_AND_MINUS_WORDS,
                            AdvqDefectTypes.bothQuotesAndMinusWords())
                    .build();

    public static final String KEYWORDS_KEY = "Keywords";
    public static final String REGIONIDS_KEY = "RegionIds";

    private static final int MIN_WORDS_PER_TRAFFIC_FORECAST = 1;
    private static final int MAX_WORDS_PER_TRAFFIC_FORECAST = 10000;

    private static final int NUM_OF_DEVICE_OPTIONS = 4;

    private static final Logger logger = LoggerFactory.getLogger(HasSearchVolumeDelegate.class);

    private final AdvqClient advq;
    private final RegionIdsValidator regionsValidator;
    private final GeoTreeFactory geoTreeFactory;
    private final ResultConverter resultConverter;
    private final StopWordService stopWordService;
    private final KeywordWithLemmasFactory keywordFactory;

    @Autowired
    public HasSearchVolumeDelegate(AdvqClient advq,
                                   GeoTreeFactory geoTreeFactory,
                                   RegionIdsValidator regionsValidator,
                                   ResultConverter resultConverter,
                                   ApiAuthenticationSource auth,
                                   StopWordService stopWordService,
                                   KeywordWithLemmasFactory keywordFactory) {
        super(ApiPathConverter.forKeywordsResearch(), auth);

        this.advq = advq;
        this.regionsValidator = regionsValidator;
        this.geoTreeFactory = geoTreeFactory;
        this.resultConverter = resultConverter;
        this.stopWordService = stopWordService;
        this.keywordFactory = keywordFactory;
    }

    @Override
    public HasSearchVolumeInnerRequest convertRequest(HasSearchVolumeRequest externalRequest) {
        logger.info("Parse request {}", externalRequest);
        return new HasSearchVolumeInnerRequest(
                mapList(externalRequest.getSelectionCriteria().getKeywords(), String::trim),
                externalRequest.getSelectionCriteria().getRegionIds(),
                EnumSet.copyOf(externalRequest.getFieldNames()));
    }

    @Override
    public ApiResult<List<KeywordSearchVolumes>> processRequest(HasSearchVolumeInnerRequest internalRequest) {
        logger.debug("processRequest {}", internalRequest);

        ValidationResult<HasSearchVolumeInnerRequest, DefectType> validation = validate(internalRequest);
        if (validation.hasAnyErrors()) {
            return ApiResult.broken(validation.flattenErrors(),
                    validation.flattenWarnings());
        }

        List<KeywordSearchVolumes> results = queryAdvq(internalRequest);
        return ApiResult.successful(
                results, validation.flattenWarnings());
    }

    @Override
    public HasSearchVolumeResponse convertResponse(ApiResult<List<KeywordSearchVolumes>> result) {
        return new HasSearchVolumeResponse()
                .withHasSearchVolumeResults(mapList(result.getResult(), this::convertItemResponse));
    }


    private ValidationResult<HasSearchVolumeInnerRequest, DefectType> validate(
            HasSearchVolumeInnerRequest request) {
        ItemValidationBuilder<HasSearchVolumeInnerRequest, DefectType> builder =
                ItemValidationBuilder.of(request);
        builder.list(request.getKeywords(), KEYWORDS_KEY)
                .check(notNull(), requiredButEmpty())
                .check(minListSize(MIN_WORDS_PER_TRAFFIC_FORECAST), absent(), When.isValid())
                .check(maxListSize(MAX_WORDS_PER_TRAFFIC_FORECAST),
                        maxElementsPerRequest(MAX_WORDS_PER_TRAFFIC_FORECAST), When.isValid())
                .checkEach(unique(), duplicatedElement(), When.isValid())
                .checkEachBy(keyword -> resultConverter.convertValidationResult(
                        new PhraseSyntaxValidator().apply(keyword), HAS_SEARCH_VOLUME_CUSTOM_DEFECT_PRESENTATIONS),
                        When.isValid())
                .checkEachBy(keyword -> resultConverter.convertValidationResult(
                        new PhraseValidator(stopWordService, keywordFactory, parseWithMinuses(keyword)).apply(keyword),
                        HAS_SEARCH_VOLUME_CUSTOM_DEFECT_PRESENTATIONS), When.isValid());
        builder.item(request.getRegionIds(), REGIONIDS_KEY)
                .check(notNull(), requiredButEmpty())
                .checkBy(regionIds -> resultConverter.convertValidationResult(regionsValidator.apply(regionIds, getGeoTree())),
                        When.isValid());

        return builder.getResult();
    }

    private List<KeywordSearchVolumes> queryAdvq(HasSearchVolumeInnerRequest request) {
        List<String> devices = filterList(
                mapList(request.getFields(), HasSearchVolumeDelegate::convertToDevice),
                Objects::nonNull);
        // Оптимизация - если приходят все четыре поля, то поле "все девайсы" откидываем, буудем его вычислять
        // как телефон + планшет + десктоп, сэкономим один запрос в advq
        if (devices.size() == NUM_OF_DEVICE_OPTIONS) {
            devices = asList(DEVICES_PHONE, DEVICES_TABLET, DEVICES_DESKTOP);
        }
        CheckMinHitsResult result = advq.checkMinHits(request.getKeywords(), request.getRegionIds(), devices);

        if (!result.isSuccessful()) {
            logger.error("Internal errors {} while processing request {}", result.getErrors(), request);
            // Мы не можем переводить ошибки advq пользователям, плюс в advq в принципе две ошибки - кодировка и
            // синтаксис ключевых слов, кодировка не должна выстрелить никогда с UTF-8, синтаксис должен отсекаться
            // валидацией
            throw new RuntimeServiceException("Advq checkMinHits error", result.getErrors().get(0));
        }
        return parseResults(result, request);
    }

    private HasSearchVolumeItem convertItemResponse(KeywordSearchVolumes item) {
        return new HasSearchVolumeItem()
                .withKeyword(item.getKeyword())
                .withRegionIds(item.getRegionIds())
                .withAllDevices(item.getAllDevices())
                .withMobilePhones(item.getMobilePhones())
                .withTablets(item.getTablets())
                .withDesktops(item.getDesktops());
    }

    private static List<KeywordSearchVolumes> parseResults(CheckMinHitsResult results,
                                                           HasSearchVolumeInnerRequest request) {
        Set<HasSearchVolumeFieldEnum> fields = request.getFields();
        Map<String, KeywordSearchVolumes> keywordsMap = request.getKeywords().stream().collect(
                toMap(String::trim, kw -> KeywordSearchVolumes.from(kw, request.getRegionIds(), fields)));

        // Заполняем поля impressions*
        results.getResults().forEach((device, deviceResult) ->
                deviceResult.forEach((keyword, keywordResult) -> {
                    YesNoEnum impressions = hasSearchVolume(keywordResult);
                    KeywordSearchVolumes target = keywordsMap.get(keyword);
                    // Заполняем поле ALL_DEVICES по принципу - если в другом поле YES, то и в нем YES
                    if (fields.contains(ALL_DEVICES) && impressions == YesNoEnum.YES) {
                        target.setAllDevices(YesNoEnum.YES);
                    }

                    switch (device) { // поле DEVICES_ALL уже вычислено
                        case DEVICES_PHONE:
                            target.setMobilePhones(impressions);
                            break;
                        case DEVICES_TABLET:
                            target.setTablets(impressions);
                            break;
                        case DEVICES_DESKTOP:
                            target.setDesktops(impressions);
                            break;
                    }
                })
        );

        return mapList(request.getKeywords(), keywordsMap::get);
    }

    /**
     * Вычисляет, были ли показы на ключевой фразе из ответа ручки ADVQ checkMinHits
     */
    private static YesNoEnum hasSearchVolume(CheckMinHitsItem checkMinHits) {
        return checkMinHits.getStat().isHitsGtMinHits() ? YesNoEnum.YES : YesNoEnum.NO;
    }

    @Nullable
    private static String convertToDevice(HasSearchVolumeFieldEnum field) {
        switch (field) {
            case ALL_DEVICES:
                return DEVICES_ALL;

            case MOBILE_PHONES:
                return DEVICES_PHONE;

            case TABLETS:
                return DEVICES_TABLET;

            case DESKTOPS:
                return DEVICES_DESKTOP;

            default:
                return null;
        }
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getApiGeoTree();
    }
}
