package ru.yandex.direct.api.v5.entity.leads.delegate;

import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.leads.GetRequest;
import com.yandex.direct.api.v5.leads.GetResponse;
import com.yandex.direct.api.v5.leads.LeadDataItem;
import com.yandex.direct.api.v5.leads.LeadFieldEnum;
import com.yandex.direct.api.v5.leads.LeadGetItem;
import com.yandex.direct.api.v5.leads.LeadsSelectionCriteria;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.leads.LeadsEndpoint;
import ru.yandex.direct.api.v5.entity.leads.LeadsServiceException;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.CommonTranslations;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.display.landing.client.DisplayLandingClient;
import ru.yandex.direct.display.landing.client.submissions.GetSubmissionsRequest;
import ru.yandex.direct.display.landing.client.submissions.GetSubmissionsResponse;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.validIso8601DateTime;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class GetLeadsDelegate extends GetApiServiceDelegate<
        GetRequest,
        GetResponse,
        LeadFieldEnum,
        GetSubmissionsRequest.SelectionCriteria,
        GetSubmissionsResponse.Submission> {
    private final DisplayLandingClient client;
    private final EnumPropertyFilter<LeadFieldEnum> propertyFilter;

    public GetLeadsDelegate(DisplayLandingClient client,
                            ApiAuthenticationSource auth,
                            PropertyFilter propertyFilter) {
        super(LeadsEndpoint.PATH_CONVERTER, auth);

        this.client = client;
        this.propertyFilter = EnumPropertyFilter.from(LeadFieldEnum.class, propertyFilter);
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);

        vb.checkBy(GetRequestGeneralValidator::validateRequest);

        vb.item(externalRequest.getSelectionCriteria(), "SelectionCriteria")
                .checkBy(this::validateSelectionCriteria, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<LeadsSelectionCriteria, DefectType> validateSelectionCriteria(
            LeadsSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<LeadsSelectionCriteria, DefectType> vb = ItemValidationBuilder.of(selectionCriteria);

        vb.item(selectionCriteria.getTurboPageIds(), "TurboPageIds")
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());

        vb.item(selectionCriteria.getDateTimeFrom(), "DateTimeFrom")
                .check(validIso8601DateTime(), When.notNull());

        vb.item(selectionCriteria.getDateTimeTo(), "DateTimeTo")
                .check(validIso8601DateTime(), When.notNull());

        return vb.getResult();
    }


    @Override
    public Set<LeadFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public GetSubmissionsRequest.SelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        LeadsSelectionCriteria external = externalRequest.getSelectionCriteria();

        GetSubmissionsRequest.SelectionCriteria internal = new GetSubmissionsRequest.SelectionCriteria()
                .withClientId(auth.getChiefSubclient().getClientId().asLong())
                .withTurboPageIds(external.getTurboPageIds());

        if (external.getDateTimeFrom() != null) {
            internal.withDateTimeFrom(OffsetDateTime.parse(external.getDateTimeFrom()));
        }

        if (external.getDateTimeTo() != null) {
            internal.withDateTimeTo(OffsetDateTime.parse(external.getDateTimeTo()));
        }

        return internal;
    }

    @Override
    public List<GetSubmissionsResponse.Submission> get(
            GenericGetRequest<LeadFieldEnum, GetSubmissionsRequest.SelectionCriteria> getRequest) {
        GetSubmissionsRequest request = new GetSubmissionsRequest()
                .withSelectionCriteria(getRequest.getSelectionCriteria())
                .withLimit(getRequest.getLimitOffset().limit())
                .withOffset(getRequest.getLimitOffset().offset());

        GetSubmissionsResponse response = client.getSubmissions(request);
        if (response.getErrorDetail() != null) {
            if (response.getErrorCode() == 53) {
                throw new LeadsServiceException("Errors while calling DisplayLandingClient",
                        54, SecurityTranslations.INSTANCE.accessDenied(), null);
            }

            throw new LeadsServiceException("Unknown error",
                    1000, CommonTranslations.INSTANCE.serviceInternalError(), null);
        }

        return response.getSubmissions();
    }

    @Override
    public GetResponse convertGetResponse(List<GetSubmissionsResponse.Submission> result,
                                          Set<LeadFieldEnum> requestedFields, @Nullable Long limitedBy) {
        List<LeadGetItem> items = mapList(result, this::convertResultItem);
        propertyFilter.filterProperties(items, requestedFields);

        return new GetResponse()
                .withLeads(items)
                .withLimitedBy(limitedBy);
    }

    private LeadGetItem convertResultItem(GetSubmissionsResponse.Submission item) {
        return new LeadGetItem()
                .withId(item.getId())
                .withTurboPageId(item.getTurboPageId())
                .withTurboPageName(item.getTurboPageName())
                .withSubmittedAt(item.getSubmittedAt().format(DateTimeFormatter.ISO_OFFSET_DATE_TIME))
                .withData(mapList(item.getData(), this::convertDataItem));
    }

    private LeadDataItem convertDataItem(GetSubmissionsResponse.DataItem item) {
        return new LeadDataItem()
                .withName(item.getFieldName())
                .withValue(item.getValue());
    }
}
