package ru.yandex.direct.api.v5.entity.negativekeywordsharedsets.delegate;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.negativekeywordsharedsets.UpdateRequest;
import com.yandex.direct.api.v5.negativekeywordsharedsets.UpdateResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.entity.negativekeywordsharedsets.converter.NegativeKeywordSharedSetsConverter;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator;
import ru.yandex.direct.core.entity.minuskeywordspack.container.UpdatedMinusKeywordsPackInfo;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.service.MinusKeywordsPacksUpdateOperationFactory;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.utils.converter.Converter;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.api.v5.validation.DefectTypes.duplicatedObject;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.unique;
import static ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.MAX_LIBRARY_PACKS_COUNT;
import static ru.yandex.direct.result.ResultConverters.massResultValueConverter;
import static ru.yandex.direct.utils.converter.Converters.nullSafeConverter;

@Service
@ParametersAreNonnullByDefault
public class UpdateNegativeKeywordSharedSetsDelegate extends OperationOnListDelegate<UpdateRequest, UpdateResponse,
        ModelChanges<MinusKeywordsPack>, Long> {

    private MinusKeywordsPacksUpdateOperationFactory minusKeywordsPacksUpdateOperationFactory;
    private ShardHelper shardHelper;
    private ResultConverter resultConverter;

    @Autowired
    public UpdateNegativeKeywordSharedSetsDelegate(
            ApiAuthenticationSource apiAuthenticationSource,
            MinusKeywordsPacksUpdateOperationFactory minusKeywordsPacksUpdateOperationFactory,
            ShardHelper shardHelper, ResultConverter resultConverter) {
        super(ApiPathConverter.forNegativeKeywordSharedSets(), apiAuthenticationSource);
        this.minusKeywordsPacksUpdateOperationFactory = minusKeywordsPacksUpdateOperationFactory;
        this.shardHelper = shardHelper;
        this.resultConverter = resultConverter;
    }

    @Nullable
    @Override
    public ValidationResult<UpdateRequest, DefectType> validateRequest(UpdateRequest externalRequest) {
        ItemValidationBuilder<UpdateRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.list(externalRequest.getNegativeKeywordSharedSets(),
                UpdateRequest.PropInfo.NEGATIVE_KEYWORD_SHARED_SETS.propertyName)
                .checkEach(notNull())
                .check(maxListSize(MAX_LIBRARY_PACKS_COUNT), maxElementsPerRequest(MAX_LIBRARY_PACKS_COUNT));
        return vb.getResult();
    }

    @Nonnull
    @Override
    public ValidationResult<List<ModelChanges<MinusKeywordsPack>>, DefectType> validateInternalRequest(List<ModelChanges<MinusKeywordsPack>> internalRequest) {
        return ListValidationBuilder.<ModelChanges<MinusKeywordsPack>, DefectType>of(internalRequest)
                .checkEach(unique(ModelChanges::getId), duplicatedObject(), When.isValid())
                .getResult();
    }

    @Override
    public List<ModelChanges<MinusKeywordsPack>> convertRequest(UpdateRequest externalRequest) {
        return externalRequest.getNegativeKeywordSharedSets()
                .stream()
                .map(NegativeKeywordSharedSetsConverter::convertUpdateRequestToModelChanges)
                .collect(toList());
    }

    @Override
    public ApiMassResult<Long> processList(List<ModelChanges<MinusKeywordsPack>> changes) {
        ClientId clientId = auth.getChiefSubclient().getClientId();

        MassResult<UpdatedMinusKeywordsPackInfo> packInfoMassResult =
                minusKeywordsPacksUpdateOperationFactory.newInstance(
                        Applicability.PARTIAL,
                        changes,
                        MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE,
                        clientId,
                        shardHelper.getShardByClientId(clientId)
                ).prepareAndApply();

        Converter<UpdatedMinusKeywordsPackInfo, Long> converter =
                nullSafeConverter(UpdatedMinusKeywordsPackInfo::getId);
        MassResult<Long> idsMassResult = massResultValueConverter(converter).convert(packInfoMassResult);

        return resultConverter.toApiMassResult(idsMassResult);
    }

    @Override
    public UpdateResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new UpdateResponse().withUpdateResults(resultConverter.toActionResults(result, apiPathConverter));
    }
}
