package ru.yandex.direct.api.v5.entity.promotedcontent.delegate;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.promotedcontent.AddRequest;
import com.yandex.direct.api.v5.promotedcontent.AddResponse;
import com.yandex.direct.api.v5.promotedcontent.AddTypeEnum;
import com.yandex.direct.api.v5.promotedcontent.PromotedContentAddItem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.OperationOnListDelegate;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionService;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionSingleObjectRequest;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.common.ApiPathConverter.forPromotedContent;
import static ru.yandex.direct.api.v5.entity.promotedcontent.Constants.MAX_COLLECTION_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.promotedcontent.Constants.MAX_EDA_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.promotedcontent.Constants.MAX_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.promotedcontent.Constants.MAX_SERVICE_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.promotedcontent.Constants.MAX_VIDEO_ELEMENTS_PER_ADD;
import static ru.yandex.direct.api.v5.entity.promotedcontent.PromotedContentDefectTypes.maxContentPerAddRequest;
import static ru.yandex.direct.api.v5.entity.promotedcontent.PromotedContentDefectTypes.wrongContentType;
import static ru.yandex.direct.api.v5.entity.promotedcontent.PromotedContentEndpoint.DEFECT_PRESENTATIONS;
import static ru.yandex.direct.api.v5.entity.promotedcontent.converter.ContentTypeConverter.convertAddTypeToCoreType;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notBlank;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;

@Component
@ParametersAreNonnullByDefault
public class AddContentDelegate extends OperationOnListDelegate<AddRequest, AddResponse,
        ContentPromotionSingleObjectRequest, Long> {

    private final ContentPromotionService contentPromotionService;
    private final ResultConverter resultConverter;
    private final ApiContextHolder apiContextHolder;

    @Autowired
    public AddContentDelegate(
            ApiAuthenticationSource auth,
            ResultConverter resultConverter,
            ApiContextHolder apiContextHolder,
            ContentPromotionService contentPromotionService) {
        super(forPromotedContent(), auth);
        this.resultConverter = resultConverter;
        this.contentPromotionService = contentPromotionService;
        this.apiContextHolder = apiContextHolder;
    }

    @Override
    public ValidationResult<AddRequest, DefectType> validateRequest(AddRequest externalRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        Map<AddTypeEnum, Long> contentSizeByType = externalRequest.getPromotedContent().stream()
                .filter(c -> c != null && c.getType() != null) // валидация на type == null будет в validateInternalRequest
                .collect(Collectors.groupingBy(PromotedContentAddItem::getType, Collectors.counting()));

        vb.item(externalRequest.getPromotedContent(), AddRequest.PropInfo.PROMOTED_CONTENT.propertyName)
                .check(eachNotNull())
                .check(maxListSize(MAX_ELEMENTS_PER_ADD), maxContentPerAddRequest(MAX_ELEMENTS_PER_ADD))
                .checkByFunction(list -> maxContentByType(AddTypeEnum.SERVICE, MAX_SERVICE_ELEMENTS_PER_ADD)
                        .apply(contentSizeByType.get(AddTypeEnum.SERVICE)))
                .checkByFunction(list -> maxContentByType(AddTypeEnum.VIDEO, MAX_VIDEO_ELEMENTS_PER_ADD)
                        .apply(contentSizeByType.get(AddTypeEnum.VIDEO)))
                .checkByFunction(list -> maxContentByType(AddTypeEnum.COLLECTION, MAX_COLLECTION_ELEMENTS_PER_ADD)
                        .apply(contentSizeByType.get(AddTypeEnum.COLLECTION)))
                .checkByFunction(list -> maxContentByType(AddTypeEnum.EDA, MAX_EDA_ELEMENTS_PER_ADD)
                        .apply(contentSizeByType.get(AddTypeEnum.EDA)));
        return vb.getResult();
    }

    private static Constraint<Long, DefectType> maxContentByType(AddTypeEnum type, int max) {
        return Constraint.fromPredicate(size -> size >= 0 && size <= max, maxContentPerAddRequest(max, type));
    }

    @Override
    @Nonnull
    public ValidationResult<List<ContentPromotionSingleObjectRequest>, DefectType> validateInternalRequest(
            List<ContentPromotionSingleObjectRequest> internalRequest) {
        boolean isServicesApp = getAuth().isServicesApplication();
        return ListValidationBuilder.<ContentPromotionSingleObjectRequest, DefectType>of(internalRequest)
                .checkEachBy(request-> {
                    var vb = ItemValidationBuilder.of(request, DefectType.class);
                    vb.item(request.getContentType(), PromotedContentAddItem.PropInfo.TYPE.propertyName)
                            .check(fromPredicateOfNullable(
                                    t -> t != null && (t != ContentPromotionContentType.SERVICE || isServicesApp),
                                    wrongContentType()));
                    vb.item(request.getUrl(), PromotedContentAddItem.PropInfo.URL.propertyName)
                            .check(notNull())
                            .check(notBlank());
                    return vb.getResult();
                })
                .getResult();
    }

    @Override
    public List<ContentPromotionSingleObjectRequest> convertRequest(AddRequest externalRequest) {
        return mapList(externalRequest.getPromotedContent(), contentAddItem -> new ContentPromotionSingleObjectRequest()
                .withUrl(contentAddItem.getUrl())
                .withContentType(convertAddTypeToCoreType(contentAddItem.getType()))
                .withRequestId(nvl(
                        ifNotNull(apiContextHolder.get().getApiLogRecord(), rec -> String.valueOf(rec.getRequestId())),
                        "0"))
                .withAdGroupId(0L)
                .withCampaignId(0L));
    }

    @Override
    public ApiMassResult<Long> processList(List<ContentPromotionSingleObjectRequest> validItems) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        fillClientLogin(validItems, auth.getChiefSubclient().getLogin());
        var result = contentPromotionService.addContentPromotion(clientId, validItems);
        return resultConverter.toApiMassResult(MassResult.successfulMassAction(result.getValue(), result),
                DEFECT_PRESENTATIONS);
    }

    private void fillClientLogin(List<ContentPromotionSingleObjectRequest> items, String clientLogin) {
        items.forEach(t -> t.withClientLogin(clientLogin));
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new AddResponse().withAddResults(resultConverter.toActionResults(result, apiPathConverter));
    }
}
