package ru.yandex.direct.api.v5.entity.promotedcontent.delegate;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.promotedcontent.AddTypeEnum;
import com.yandex.direct.api.v5.promotedcontent.GetRequest;
import com.yandex.direct.api.v5.promotedcontent.GetResponse;
import com.yandex.direct.api.v5.promotedcontent.PromotedContentFieldEnum;
import com.yandex.direct.api.v5.promotedcontent.PromotedContentGetItem;
import com.yandex.direct.api.v5.promotedcontent.PromotedContentSelectionCriteria;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.promotedcontent.converter.ContentTypeConverter;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContent;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.entity.contentpromotion.repository.ContentPromotionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.yandex.direct.api.v5.ads.GetRequest.PropInfo.SELECTION_CRITERIA;
import static java.util.Arrays.asList;
import static ru.yandex.direct.api.v5.common.ApiPathConverter.forPromotedContent;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.entity.promotedcontent.PromotedContentDefectTypes.wrongContentType;
import static ru.yandex.direct.api.v5.entity.promotedcontent.converter.ContentTypeConverter.convertCoreTypeToGetType;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.missedParamsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateOfNullable;

@Component
@ParametersAreNonnullByDefault
public class GetContentDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, PromotedContentFieldEnum,
        PromotedContentSelectionCriteria, ContentPromotionContent> {

    private final ContentPromotionRepository contentPromotionRepository;
    private final EnumPropertyFilter<PromotedContentFieldEnum> propertyFilter;

    @Autowired
    public GetContentDelegate(ApiAuthenticationSource auth,
                              ContentPromotionRepository contentPromotionRepository,
                              PropertyFilter propertyFilter) {
        super(forPromotedContent(), auth);
        this.contentPromotionRepository = contentPromotionRepository;
        this.propertyFilter = EnumPropertyFilter.from(PromotedContentFieldEnum.class, propertyFilter);
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getSelectionCriteria(), SELECTION_CRITERIA.schemaName.getLocalPart())
                .check(notNull())
                .checkBy(this::validateSelectionCriteria);
        return vb.getResult();
    }

    /**
     * Валидируем, что что-то разумное указали в selection_criteria: либо непустой список типов, либо непустой список id
     * Валидируем длину списка id
     * Валидируем, что типы непустые
     */
    private ValidationResult<PromotedContentSelectionCriteria, DefectType> validateSelectionCriteria(
            PromotedContentSelectionCriteria selectionCriteria) {
        var selectionVb = ItemValidationBuilder.<PromotedContentSelectionCriteria, DefectType>of(selectionCriteria);
        selectionVb.check(fromPredicate(sc -> !sc.getIds().isEmpty() || !sc.getTypes().isEmpty(),
                missedParamsInSelection(String.join(
                        ", ", asList(PromotedContentSelectionCriteria.PropInfo.IDS.schemaName.getLocalPart(),
                                PromotedContentSelectionCriteria.PropInfo.TYPES.schemaName.getLocalPart())))));

        selectionVb.item(selectionCriteria.getIds(),
                PromotedContentSelectionCriteria.PropInfo.IDS.schemaName.toString())
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());

        boolean isServicesApp = getAuth().isServicesApplication();
        selectionVb.item(selectionCriteria.getTypes(), PromotedContentSelectionCriteria.PropInfo.TYPES
                .schemaName.getLocalPart())
                .checkBy(types -> ListValidationBuilder.<AddTypeEnum, DefectType>of(types)
                        .checkEach(fromPredicateOfNullable(
                                t -> t != null && (t != AddTypeEnum.SERVICE || isServicesApp),
                                wrongContentType()))
                        .getResult());
        return selectionVb.getResult();
    }

    @Override
    public Set<PromotedContentFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public PromotedContentSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        return externalRequest.getSelectionCriteria();
    }

    @Override
    public List<ContentPromotionContent> get(
            GenericGetRequest<PromotedContentFieldEnum, PromotedContentSelectionCriteria> getRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        List<ContentPromotionContent> result =
                contentPromotionRepository.getContentPromotionsByContentIdsAndTypes(clientId,
                getRequest.getSelectionCriteria().getIds(),
                listToSet(getRequest.getSelectionCriteria().getTypes(),
                        ContentTypeConverter::convertAddTypeToCoreType));

        // отфильтровываем Услуги, если к ним нет доступа
        boolean isServicesApp = getAuth().isServicesApplication();
        return StreamEx.of(result)
                .filter(content -> content.getType() != ContentPromotionContentType.SERVICE || isServicesApp)
                .toList();
    }

    @Override
    public GetResponse convertGetResponse(List<ContentPromotionContent> result,
                                          Set<PromotedContentFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        if (!result.isEmpty()) {
            List<PromotedContentGetItem> convertedItems = mapList(result, t -> new PromotedContentGetItem()
                    .withId(t.getId())
                    .withIsAvailable(t.getIsInaccessible() ? YesNoEnum.NO : YesNoEnum.YES)
                    .withType(convertCoreTypeToGetType(t.getType()))
                    .withUrl(t.getUrl()));

            propertyFilter.filterProperties(convertedItems, requestedFields);
            response.withPromotedContent(convertedItems);
        }
        return response;
    }

}
