package ru.yandex.direct.api.v5.entity.retargetinglists.converter;

import java.util.List;

import com.yandex.direct.api.v5.retargetinglists.AddRequest;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleArgumentItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleOperatorEnum;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListTypeEnum;
import org.springframework.core.convert.converter.Converter;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;

import static ru.yandex.direct.api.v5.entity.retargetinglists.converter.GoalInterestsTypeConverter.getInterestType;
import static ru.yandex.direct.api.v5.entity.retargetinglists.converter.GoalInterestsTypeConverter.getInternalId;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.INTERESTS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class AddRequestConverter implements Converter<AddRequest, List<RetargetingCondition>> {

    @Override
    public List<RetargetingCondition> convert(AddRequest addRequest) {
        return mapList(addRequest.getRetargetingLists(), userList -> {
            RetargetingCondition retargetingCondition = new RetargetingCondition();
            retargetingCondition.withName(userList.getName())
                    .withType(convertType(userList.getType()))
                    .withDescription(userList.getDescription())
                    .withRules(convertRules(userList.getRules()));
            return retargetingCondition;
        });
    }

    private ConditionType convertType(RetargetingListTypeEnum type) {
        // Тип по умолчанию - metrika_goals
        return type == RetargetingListTypeEnum.AUDIENCE ? ConditionType.interests : ConditionType.metrika_goals;
    }

    List<Rule> convertRules(List<RetargetingListRuleItem> rules) {
        return mapList(rules, item -> {
            Rule rule = new Rule();

            // Тип интереса вычисляем по первой цели с типом INTERESTS.
            // В RulesAddUpdateRequestValidator есть проверка, что все типы интересов в правиле совпадают
            CryptaInterestType interestType = item.getArguments().stream()
                    .filter(arg -> Goal.computeType(getInternalId(arg.getExternalId())) == INTERESTS)
                    .map(arg -> getInterestType(arg.getExternalId()))
                    .findFirst()
                    .orElse(null);

            rule.withGoals(convertArguments(item.getArguments()))
                    .withType(convertOperator(item.getOperator()))
                    .withInterestType(interestType);
            return rule;
        });
    }

    private List<Goal> convertArguments(List<RetargetingListRuleArgumentItem> arguments) {
        return mapList(arguments, arg -> {

            // Там где из префикса получилось определить длительность интереса, этот префикс нужно удалить
            // Смотри описание в {@link GoalInterestsTypeConverter}
            Long id = getInterestType(arg.getExternalId()) == null
                    ? arg.getExternalId() : getInternalId(arg.getExternalId());

            Goal goal = new Goal();
            goal.withId(id)
                    .withTime(arg.getMembershipLifeSpan());
            return goal;
        });
    }

    private RuleType convertOperator(RetargetingListRuleOperatorEnum operator) {
        if (operator == null) {
            return null;
        }
        switch (operator) {
            case ALL:
                return RuleType.ALL;
            case ANY:
                return RuleType.OR;
            case NONE:
                return RuleType.NOT;
            default:
                throw new IllegalStateException("unknown operator type: " + operator.toString());
        }
    }
}
