package ru.yandex.direct.api.v5.entity.retargetinglists.converter;

import com.yandex.direct.api.v5.dictionaries.InterestTypeEnum;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;

import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.all;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.long_term;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.short_term;

/**
 * Для интересов (таблица crypta_goals, где crypta_goal_type='interests') нужно задавать длительность (короткий интерес,
 * длинный или все). Для этого перед идентификатором цели устанавливается префикс:
 * 10 - короткий интерес
 * 20 - длинный интерес
 * 30 -все
 * <p>
 * При получении внутреннего идентификатора префикс отбрасывается, при получении внешнего - добавляется
 */
@Component
public class GoalInterestsTypeConverter {
    public static final long MAX_ID_FOR_PREFIX = 10_000_000_000L;
    public static final int SHORT_TERM_PREFIX = 10;
    public static final int LONG_TERM_PREFIX = 20;
    public static final int ANY_TERM_PREFIX = 30;

    private GoalInterestsTypeConverter() {
    }

    //Для цели с типом INTERESTS отбрасываем префикс
    public static Long getInternalId(Long externalId) {

        long id = externalId % MAX_ID_FOR_PREFIX;
        if (Goal.computeType(id) == GoalType.INTERESTS) {
            return id;
        }

        return externalId;
    }

    public static Long getExternalId(Long id, CryptaInterestType interestType) {
        if (Goal.computeType(id) == GoalType.INTERESTS) {
            int prefix;
            switch (interestType) {
                case short_term:
                    prefix = SHORT_TERM_PREFIX;
                    break;
                case long_term:
                    prefix = LONG_TERM_PREFIX;
                    break;
                case all:
                    prefix = ANY_TERM_PREFIX;
                    break;
                default:
                    throw new IllegalArgumentException("Wrong interest type " + interestType);
            }
            return prefix * MAX_ID_FOR_PREFIX + id;
        }

        return id;
    }

    //Для цели с типом INTERESTS по префиксу определяем длительность интереса
    public static CryptaInterestType getInterestType(Long externalId) {

        if (Goal.computeType(externalId % MAX_ID_FOR_PREFIX) == GoalType.INTERESTS) {

            int prefix = (int) (externalId / MAX_ID_FOR_PREFIX);

            switch (prefix) {
                case SHORT_TERM_PREFIX:
                    return short_term;
                case LONG_TERM_PREFIX:
                    return long_term;
                case ANY_TERM_PREFIX:
                    return all;
            }
        }

        return null;
    }

    public static InterestTypeEnum getExternalType(CryptaInterestType interestType) {
        switch (interestType) {
            case short_term:
                return InterestTypeEnum.SHORT_TERM;
            case long_term:
                return InterestTypeEnum.LONG_TERM;
            case all:
                return InterestTypeEnum.ANY;
            default:
                throw new IllegalArgumentException("Wrong interest type " + interestType);
        }
    }
}
