package ru.yandex.direct.api.v5.entity.retargetinglists.delegate;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.retargetinglists.GetRequest;
import com.yandex.direct.api.v5.retargetinglists.GetResponse;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListFieldEnum;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListGetItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListSelectionCriteria;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.ApiPathConverter;
import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.retargetinglists.converter.GetRequestConverter;
import ru.yandex.direct.api.v5.entity.retargetinglists.converter.RetargetingConditionConverter;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.client.service.checker.ClientAccessCheckerTypeSupportFacade;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelectionCriteria;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class GetRetargetingListsDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, RetargetingListFieldEnum, RetargetingSelectionCriteria,
                RetargetingCondition> {

    private final RetargetingConditionService retConditionService;
    private final EnumPropertyFilter<RetargetingListFieldEnum> propertyFilter;
    private final RetargetingConditionConverter retConditionConverter;
    private final ResultConverter resultConverter;
    private final GetRequestConverter getRequestConverter;
    private final ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade;
    private final FeatureService featureService;

    @Autowired
    public GetRetargetingListsDelegate(ApiAuthenticationSource auth,
                                       RetargetingConditionService retConditionService,
                                       PropertyFilter propertyFilter,
                                       RetargetingConditionConverter retConditionConverter,
                                       ResultConverter resultConverter,
                                       GetRequestConverter getRequestConverter,
                                       ClientAccessCheckerTypeSupportFacade clientAccessCheckerTypeSupportFacade,
                                       FeatureService featureService) {
        super(ApiPathConverter.forRetargetingLists(), auth);
        this.retConditionService = retConditionService;
        this.propertyFilter = EnumPropertyFilter.from(RetargetingListFieldEnum.class, propertyFilter);
        this.retConditionConverter = retConditionConverter;
        this.resultConverter = resultConverter;
        this.getRequestConverter = getRequestConverter;
        this.clientAccessCheckerTypeSupportFacade = clientAccessCheckerTypeSupportFacade;
        this.featureService = featureService;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ModelItemValidationBuilder
                .of(externalRequest);

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes);

        vb.item(externalRequest.getSelectionCriteria(), "SelectionCriteria")
                .checkBy(this::validateSelectionCriteria, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<RetargetingListSelectionCriteria, DefectType> validateSelectionCriteria(
            RetargetingListSelectionCriteria selectionCriteria) {
        ItemValidationBuilder<RetargetingListSelectionCriteria, DefectType> vb =
                ItemValidationBuilder.of(selectionCriteria);

        vb.item(selectionCriteria.getIds(), "Ids")
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection());

        return vb.getResult();
    }

    @Override
    public RetargetingSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        return getRequestConverter.convert(externalRequest);
    }

    @Override
    public Set<RetargetingListFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public List<RetargetingCondition> get(
            GenericGetRequest<RetargetingListFieldEnum, RetargetingSelectionCriteria> getRequest) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        RetargetingSelectionCriteria selectionCriteria = getRequest.getSelectionCriteria();
        Set<Long> ids = selectionCriteria.getIds();
        if (ids != null && !ids.isEmpty()) {
            ids = clientAccessCheckerTypeSupportFacade.sendToCheck(
                    Map.of(RetargetingCondition.class, selectionCriteria.getIds()),
                    clientId)
                    .getOrDefault(RetargetingCondition.class, Collections.emptySet());
            if (ids.isEmpty()) {
                return Collections.emptyList();
            }
        }
        return retConditionService.getRetargetingConditions(clientId, ids,
                selectionCriteria.getTypes(), getRequest.getLimitOffset());
    }

    @Override
    public GetResponse convertGetResponse(List<RetargetingCondition> result,
                                          Set<RetargetingListFieldEnum> requestedFields, @Nullable Long limitedBy) {
        GetResponse response = new GetResponse().withLimitedBy(limitedBy);

        Set<String> clientFeatures = featureService.getEnabledForClientId(auth.getChiefSubclient().getClientId());

        AllowedRetargetingComponentsInUserProfile allowedComponents = new AllowedRetargetingComponentsInUserProfile()
                .withSocialDemoInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_SOCIAL_DEMO_IN_USER_PROFILE.getName()))
                .withFamilyAndBehaviorsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_FAMILY_AND_BEHAVIORS_IN_USER_PROFILE.getName()))
                .withAllInterestsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_ALL_INTERESTS_IN_USER_PROFILE.getName()))
                .withMetrikaInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_METRIKA_AND_AUDIENCE_IN_USER_PROFILE.getName()));

        List<RetargetingListGetItem> getItems = mapList(result, item -> retConditionConverter.convert(item,
                allowedComponents));
        propertyFilter.filterProperties(getItems, requestedFields);
        if (!getItems.isEmpty()) {
            response.withRetargetingLists(getItems);
        }
        return response;
    }
}
